%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
% Technical Note TN139
% Parameters for 'Motor Testbench Rotor Field-Oriented Control (RFOC)'
% Copyright imperix ltd, 2023
% dev@imperix.ch
%
% Revision history:
% 12.05.2022 : first version
% 14.07.2023 : "Res.p" set to 1 pair of poles (instead of 2) following a
%              bug fix in the ACG SDK (>= v3.8.1).
%              Direction of rotation changed to "Clockwise" in the RES
%              block to follow the IEC 60034-8 standard. Motor testbenches
%              shipped prior to July 2023 were physically wired with the
%              reverse convention (a clockwise rotation of the shaft would
%              decrease the position angle).
%              Finally, "Pmsm.Phi_pm" was corrected to 0.2748 Wb.
%
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

%% =================== System parameters ======================== %%

ControlFreq    = 20e3;              % Control interrupt frequency [Hz]
ControlPeriod  = 1/ControlFreq;     % Control interrupt period [s]
SamplingPeriod = ControlPeriod;     % Sampling period [s]

% Deadtime
dt = 500e-9;

%% =================== Plant model ======================== %%

% Induction machine (IM)
% SEW Eurodrive DRN112M4
IM.p = 2;               % Nb of pole pair [-]
IM.Usn = 230;           % Rated phase voltage [V]
IM.Isn = 8.4;           % Rated current [A]
IM.fsn = 50;            % Rated stator frequency [Hz]
IM.wsn = 2*pi*IM.fsn;   % Rated angular frequency [rad/s]
IM.Wrn = 1464;          % Nominal rotor speed [rpm]
IM.Rs = 1.24;           % Stator resistance [Ohm]
IM.Rr = 0.73;           % Rotor resistance [Ohm]
IM.Jm = 178e-4;         % Moment of inertia [kg m^2]
IM.Temn = 26;           % Nominal torque [Nm]

IM.cos_phi = 0.81;      % Power factor [-]
IM.sin_phi = sqrt(1 - IM.cos_phi^2);

IM.Lm = 183e-3;                     % Mutual inductance [H]
IM.Lgs = 11.5e-3;                   % Stator leakage inductance [H]
IM.Lgr = 11.5e-3;                   % Rotor leakage inductance [H]
IM.Ls = IM.Lgs + IM.Lm;             % Stator inductance [H]
IM.Lr = IM.Lgr + IM.Lm;             % Rotor inductance [H] 
IM.Lsigma = IM.Ls - IM.Lm^2/IM.Lr;  % Total leakage inductance [H]

Is_re = IM.Isn*IM.cos_phi;
Is_im = IM.Isn*IM.sin_phi;
Um_re = IM.Usn - IM.Rs*Is_re + IM.wsn*IM.Lgs*Is_im;     % Voltage across Lm (real)
Um_im = -IM.Rs*Is_im - IM.wsn*IM.Lgs*Is_re;             % Voltage across Lm (imaginary)
Um = sqrt(Um_re^2 + Um_im^2);                           % Voltage across Lm (norm)

IM.Idsn = (sqrt(2)*Um) / (IM.wsn*IM.Lm);       % Nominal d-axis current
IM.Psi_d_n = IM.Lm * IM.Idsn;                  % Nominal rotor flux

% Permanent Magnets Synchronous Machine (PMSM)
% SEW Eurodrive CM3C80L
Pmsm.p = 4;				% Pole pairs [-]
Pmsm.Usn = 230;			% Rated voltage [V]
Pmsm.Isn = 11.2;        % Rated current [Arms]
Pmsm.fsn = 50;			% Rated frequency [Hz]
Pmsm.wsn = 2*pi*50;		% Rated angular frequency [rad/s]
Pmsm.Rs = 0.559;		% Stator resistance [Ohm]
Pmsm.Ls = 4.24e-3;		% Stator inductance [H]
Pmsm.Ld = Pmsm.Ls;      % Stator inductance along d axis [H]
Pmsm.Lq = Pmsm.Ls;      % Stator inductance along q axis [H]

Pmsm.Jm = 40.6e-4;      % Moment of inertia [kg m^2]
Pmsm.kF = 0.0003035;	% Friction coefficient
Pmsm.Phi_pm = 0.2748;	% Flux from permanent magnets [Nm/Apeak]

Pmsm.Temn = 22.8; 		% Nominal torque [Nm]
Pmsm.Pn = 4800;         % Nominal active power [W]

Pmsm.Is_max = 34.9;		% Peak current [A] (in reality 34.9A)
Pmsm.Wm_Max = 2750;		% Max mechanical speed [rpm] (tbd)

Pmsm.Jm_brk = 46.2e-4;  % Inertia of the safety brake [kg m^2]

% Resolver (RH1M built into the CM3C80L)
Res.p = 1;              % Pole pairs [-]

% DC Link initial conditions
DcLink.Vdc0 = 800;      % Initial DC bus voltage [V]


%% ======================== IM - Current control ======================= %%

% Plant model
Current.K1 = 1 / IM.Rs;
Current.T1 = IM.Lsigma / IM.Rs;

% Small time constants
Current.Ts = ControlPeriod;           % Current control period [s]

Current.Tsens = 0;                    % Sensing delay [s]
Current.Tctrl = 2*Current.Ts;         % Control delay [s] (worst case scenario)
Current.Tpwm = ControlPeriod;         % Modulator delay [s]

Current.Ttot = Current.Tsens + Current.Tctrl + Current.Tpwm;
Current.Kcm = 1;

% Magnitude optimum
Current.Tn = Current.T1;
Current.Ti = 2 * Current.Kcm * Current.K1 * Current.Ttot;

Current.Kp = Current.Tn / Current.Ti;
Current.Ki = 1 / Current.Ti;


%% ======================== IM - Speed control ========================= %%

% Filter derivative of the mechanical angle
Speed.fc = 200;

% Decimation ratio of the control
Speed.Ratio = 100;

% Torque saturation
Speed.Tem_sat = 1.1 * IM.Temn;


%% =============== PMSM - Current control ================= %%

% Plant model
Ipmsm.K1 = 1 / Pmsm.Rs;
Ipmsm.T1 = Pmsm.Ld / Pmsm.Rs;
Ipmsm.K2 = 1 / Pmsm.Rs;
Ipmsm.T2 = Pmsm.Lq / Pmsm.Rs;

% Small time constants
Ipmsm.Ts = ControlPeriod;           % Current control period [s]

Ipmsm.Tsens = 0;                    % Sensing delay [s]
Ipmsm.Tctrl = 2*Ipmsm.Ts;           % Control delay [s] (worst case scenario)
Ipmsm.Tpwm = ControlPeriod;         % Modulator delay [s]

Ipmsm.Ttot = Ipmsm.Tsens + Ipmsm.Tctrl + Ipmsm.Tpwm;
Ipmsm.Kcm = 1;

% Magnitude optimum
Ipmsm.Tn_d = Ipmsm.T1;
Ipmsm.Ti_d = 2 * Ipmsm.Kcm * Ipmsm.K1 * Ipmsm.Ttot;
Ipmsm.Tn_q = Ipmsm.T2;
Ipmsm.Ti_q = 2 * Ipmsm.Kcm * Ipmsm.K2 * Ipmsm.Ttot;

Ipmsm.Kp_d = Ipmsm.Tn_d / Ipmsm.Ti_d;
Ipmsm.Ki_d = 1 / Ipmsm.Ti_d;
Ipmsm.Kp_q = Ipmsm.Tn_q / Ipmsm.Ti_q;
Ipmsm.Ki_q = 1 / Ipmsm.Ti_q;
