if {$argc != 1} {
    puts "The script requires the project name to be set as an argument."
} else {
    start_gui
    
    # Set paths
    set proj_name [lindex $argv 0]
    set script_path [file dirname [file normalize [info script]]]
    set vivado_dir_path ../vivado
    set ip_repo_dir_path ../ix_repo
    set hdl_dir_path ../hdl
    set constraint_dir_path ../constraints
    
    # Create project
    create_project $proj_name $vivado_dir_path/$proj_name -part xczu7ev-ffvc1156-2-i
    set_property target_language VHDL [current_project]
    
    # Work-around suggested by Xilinx to avoid the following inconsequential critical warning
    # [Memdata 28-165] The reference name: top_i_ip_m_blk_mem_gen_0 was not found in a previous reference definition.
    # Either the bmm file or the bmm_info_* properties are malformed, therefore BRAM INIT strings can not be populated.
    # Source: https://support.xilinx.com/s/article/59639?language=en_US
    set_param memdata.disableAnnotatedBMM 1
    set_param memdata.disableBRAMInitStrings 1
    set_param memdata.ignoreEmbeddedBMMProperties 1
    
    # Configure IP repository
    set_property ip_repo_paths $ip_repo_dir_path [current_project]
    update_ip_catalog
    
    # Create block design
    create_bd_design "top"
    
    # Add IXIP
    create_bd_cell -type ip -vlnv imperix.ch:ix:IXIP:4.0 IXIP_0
    set_property name IXIP [get_bd_cells IXIP_0]
    
    # Add constraints
    add_files -fileset constrs_1 -norecurse $constraint_dir_path/sandbox_pins_4.0rev0.xdc
    
    # Make external pins
    make_bd_pins_external [get_bd_pins IXIP/private_out]
    make_bd_pins_external [get_bd_pins IXIP/dout]
    make_bd_pins_external [get_bd_pins IXIP/private_in]
    make_bd_pins_external [get_bd_pins IXIP/din]
    make_bd_pins_external [get_bd_pins IXIP/private_io]
    make_bd_pins_external [get_bd_pins IXIP/private_io]
    make_bd_intf_pins_external [get_bd_intf_pins IXIP/USR]
    
    # Rename external ports
    set_property name private_out [get_bd_ports private_out_0]
    set_property name dout [get_bd_ports dout_0]
    set_property name private_in [get_bd_ports private_in_0]
    set_property name din [get_bd_ports din_0]
    set_property name private_io [get_bd_ports private_io_0]
    set_property name USR [get_bd_intf_ports USR_0]
    
    # Add USER_ID constant
    create_bd_cell -type ip -vlnv xilinx.com:ip:xlconstant:1.1 xlconstant_0
    set_property name USER_ID [get_bd_cells xlconstant_0]
    set_property -dict [list CONFIG.CONST_WIDTH {16}] [get_bd_cells USER_ID]
    set_property -dict [list CONFIG.CONST_VAL {1}] [get_bd_cells USER_ID]
    connect_bd_net [get_bd_pins USER_ID/dout] [get_bd_pins IXIP/user_fw_id]
    
    # Create wrapper
    make_wrapper -files [get_files $vivado_dir_path/$proj_name/$proj_name.srcs/sources_1/bd/top/top.bd] -top
    import_files -force -norecurse $vivado_dir_path/$proj_name/$proj_name.gen/sources_1/bd/top/hdl/top_wrapper.vhd
    
    # Add AXIS interface for 24 ADC channels
    add_files -norecurse $hdl_dir_path/AXIS_interface_24adc.vhd
    create_bd_cell -type module -reference ix_axis_interface_24adc ix_axis_interface_24adc_0
    set_property name ix_axis_interface_24adc [get_bd_cells ix_axis_interface_24adc_0]
    
    # Connect AXIS interface to IXIP
    connect_bd_intf_net [get_bd_intf_pins IXIP/SBIO_BUS] [get_bd_intf_pins ix_axis_interface_24adc/SBIO_BUS]
    connect_bd_intf_net [get_bd_intf_pins IXIP/ADC] [get_bd_intf_pins ix_axis_interface_24adc/ADC]
    connect_bd_net [get_bd_pins IXIP/adc_done_pulse] [get_bd_pins ix_axis_interface_24adc/adc_done_pulse_in]
    connect_bd_net [get_bd_pins IXIP/data_valid_pulse] [get_bd_pins ix_axis_interface_24adc/data_valid_pulse_in]
    connect_bd_net [get_bd_pins IXIP/reading] [get_bd_pins ix_axis_interface_24adc/data_reading_in]
    connect_bd_net [get_bd_pins IXIP/clk_250_mhz] [get_bd_pins ix_axis_interface_24adc/clk]
    connect_bd_net [get_bd_pins IXIP/sync_pulse] [get_bd_pins ix_axis_interface_24adc/sync_pulse_in]
    
    # Finalize
    regenerate_bd_layout
    save_bd_design
}