
// Central PV inverter (three-phase grid-tied) v3
// https://imperix.com/doc/example/three-phase-pv-inverter

#include "user.h"

/**
 * Specify the number of devices used by this code.
 * This information is used by Cockpit for display purposes only.
 */
NUMBER_OF_DEVICES(1);

/**
 * Global variables
 * float, int, and unsigned int, can be monitored from Cockpit
 */

float Eb;													// Boost switching point voltage
float activate = 0;											// Variable used to start and stop the converter
float core_state = 0;										// The current state of the BBOS core.

float Ipv, Ipv_ref = 0;										// Solar panel current measurement and current reference
float Vpv;													// Solar panel voltage measurement
float Vdc, Vdc_ref = 700;										// DC bus voltage measurement and voltage reference
float Ig_d_feedforward;										// Feed-forward term of the d-axis current (active power flow)

TimeDomain Vg_abc;											// Grid voltages in stationary reference frame [V]
float Vg_a, Vg_b, Vg_c;
TimeDomain Ig_abc;											// Grid currents in stationary reference frame [A]
float Ig_a, Ig_b, Ig_c;
SpaceVector Vg_dq0;											// Grid voltages in synchronous reference frame [V]
float Vg_d, Vg_q;
SpaceVector Ig_dq0, Ig_dq0_ref; 							// Grid currents (actual and reference) in synchronous reference frame [A]
float Ig_d, Ig_q, Ig_d_ref, Ig_q_ref;
SpaceVector Eg_dq0; 										// Converter EMF in synchronous RF
TimeDomain Eg_abc;											// Converter EMF in stationary reference frame
float w_grid;												// Grid angular frequency

float precharge_relay, bypass_relay;						// Relay states ('0' -> open, '1' -> closed)
float activate_inverter, activate_boost = 0;				// Activate the boost/inverter part of the converter (Used by the operation state machine)

float SubTaskTimer = 0.0;									// Timer for MPPT background loop
bool SubTaskFlag = false;									// Flag for MPPT background loop
MPPTracker mpptracker;										// MPPT for PV control current reference

DQPLLParameters DQPLL;										// DQ PLL for grid synchronization
float Theta;												// Phase angle of the grid voltage

PIDController Ipv_reg;										// Controller for the PV current control
PIDController Vdc_reg, Ig_d_reg, Ig_q_reg;					// Controllers for the DC bus voltage control

float Kp_Ipv = Tn1/Ti1, Ki_Ipv = 1/Ti1*SAMPLING_PERIOD; 	// PV current controller proportional and integral coefficient
float Kp_Ig = Tn2/Ti2, Ki_Ig = 1/Ti2*SAMPLING_PERIOD;		// Grid current controller proportional and integral coefficient
float Kp_Vdc = 0.3, Ki_Vdc = 1*SAMPLING_PERIOD;				// DC bus voltage controller proportional and integral coefficient

unsigned int err_No_GridSync, err_No_Grid, err_Vpv_low, err_Vpv_high, err_Vdc_low, err_Vdc_high;
															// Error flags

float Vdc_max = 820;										// Maximum DC bus voltage
float Vpv_min = 20;											// Minimum PV voltage
float Vpv_max = 600;										// Maximum PV voltage
float w_min = 308;											// Minimum grid angular frequency
float w_max = 320;											// Maximum grid angular frequency
float Vgrid_min = 20;										// Minimum grid voltage (phase-to-phase peak)
float Vg_rectified;											// Rectified phase-to-phase grid voltage

// Global variables in an unnamed namespace will not show up in Cockpit
namespace{unsigned int precharge_cnt, operation_cnt;};		// Counter for both state machines
bool Precharge_ready;										// Output signals of the precharge state machine
bool Precharge_fault;
tStatePrecharge State_precharge = PRECHARGE_INIT;			// Current state of the precharge state machine
tStatePrecharge Next_state_precharge = PRECHARGE_STANDBY;	// Next state of the precharge state machine
tStateOperation State_operation = OP_INIT;					// Current state of the operational state machine
tStateOperation Next_state_operation = OP_STANDBY;			// Next state of the operational state machine
unsigned int	state_precharge_uint;						// Current state of the precharge state machine for monitoring in Cockpit
unsigned int	state_operation_uint;						// Current state of the operation state machine for monitoring in Cockpit


/**
 * Functions prototypes
 */
tUserSafe UserBackground();

// Current and voltage control functions
void User_readADC();
void User_runPVCurrentCtrl();
void User_runDQPLL();
void User_runDCBusVoltageCtrl();
void User_runPVCurrentCtrl();
void User_runMPPT();
void User_setDutyCycles();

// State machines functions
void User_TestErrors();
void User_RunPrechargeFSM();
void User_RunOperationFSM();
void User_operateRelays();

// User fault function
void User_throwUserFault();

/**
 * Initialization routine executed only once, before the first call of the main interrupt
 * To be used to configure all needed peripherals and perform all needed initializations
 */
tUserSafe UserInit(void)
{
	/**
	 * Configuration of the main interrupt:
	 * - The frequency of CLOCK_0 is set to the desired switching frequency
	 * - The main interrupt routine is mapped on CLOCK_0, with a phase of 0.5.
	 * - As the the sampling of the ADC channels happens just before the interrupt is executed,
	 *   setting a phase of 0.5 allows to sample in the middle of each switching periods.
	 */
	Clock_SetFrequency(CLOCK_0, SW_FREQ);
	ConfigureMainInterrupt(UserInterrupt, CLOCK_0, 0.5);
	RegisterBackgroundCallback(UserBackground);

	/**
	 * Configuration of the ADC channels, with:
	 * - sensor sensitivity defined in user.h
	 * - no sensor offset compensation
	 */

	// Ig_abc
	Adc_ConfigureInput(ADC0, IX_PEB8038_I_GAIN/4, 0.0);
	Adc_ConfigureInput(ADC1, IX_PEB8038_I_GAIN/4, 0.0);
	Adc_ConfigureInput(ADC2, IX_PEB8038_I_GAIN/4, 0.0);
	// Ipv
	Adc_ConfigureInput(ADC3, IX_PEB8038_I_GAIN/4, 0.0);
	// Vdc
	Adc_ConfigureInput(ADC8, IX_PEB8038_V_GAIN/2, 0.0);
	// Vpv
	Adc_ConfigureInput(ADC12, IX_DIN800V_GAIN/4, 0.0);
	// Vg_abc
	Adc_ConfigureInput(ADC13, IX_DIN800V_GAIN/4, 0.0);
	Adc_ConfigureInput(ADC14, IX_DIN800V_GAIN/4, 0.0);
	Adc_ConfigureInput(ADC15, IX_DIN800V_GAIN/4, 0.0);


	/**
	 * Configuration of the dq PLL
	 */
	ConfigDQPLL(&DQPLL, 0.01, 1*SAMPLING_PERIOD, OMEGA, SAMPLING_PERIOD);

	/**
	 * Configuration of the MPPT tracker
	 */
	ConfigMPPTracker(&mpptracker, 0.1, 0, IS, 0, 1-K_IIR_LPF);

	/**
	 * Configuration of the PID controllers
	 */
	ConfigPIDController(&Ipv_reg, Kp_Ipv, Ki_Ipv, 0, 800, -800, SAMPLING_PERIOD, 10);
	ConfigPIDController(&Vdc_reg, Kp_Vdc, Ki_Vdc, 0, 40, -40, SAMPLING_PERIOD, 10);
	ConfigPIDController(&Ig_d_reg, Kp_Ig, Ki_Ig, 0, 800, -800, SAMPLING_PERIOD, 10);
	ConfigPIDController(&Ig_q_reg, Kp_Ig, Ki_Ig, 0, 800, -800, SAMPLING_PERIOD, 10);


	/**
	 * Configuration of PWM channel 0 (lines 0 & 1) to 4
	 * - mapping on CLOCK_0
	 * - triangle carrier
	 * - 1 microsecond dead-time between complementary signals
	 * - Channel 0 and 1 have a phase difference of 0.5 to interleave the two boost converters
	 */
	CbPwm_ConfigureChannel(PWM_CHANNEL_0, CLOCK_0, TRIANGLE, DEADTIME, PHASE0);
	CbPwm_ConfigureChannel(PWM_CHANNEL_1, CLOCK_0, TRIANGLE, DEADTIME, PHASE0);
	CbPwm_ConfigureChannel(PWM_CHANNEL_2, CLOCK_0, TRIANGLE, DEADTIME, PHASE0);
	CbPwm_ConfigureChannel(PWM_CHANNEL_3, CLOCK_0, TRIANGLE, DEADTIME, PHASE0);

	// Only for HIL tests
	DoPwm_ConfigureOutputMode(PWM_CHANNEL_4_H, INDEPENDENT, 0);
	DoPwm_Activate(PWM_CHANNEL_4_H, 0);
	DoPwm_ConfigureOutputMode(PWM_CHANNEL_4_L, INDEPENDENT, 0);
	DoPwm_Activate(PWM_CHANNEL_4_L, 0);
	DoPwm_ConfigureOutputMode(PWM_CHANNEL_5_H, INDEPENDENT, 0);
	DoPwm_Activate(PWM_CHANNEL_5_H, 0);

	// Log messages for state machine monitoring (severity 10 = info, 30 = error)
	Log_AddMsg(0, 10, "Precharge state: Standby");
	Log_AddMsg(1, 10, "Precharge state: Synchronizing");
	Log_AddMsg(2, 10, "Precharge state: Charging");
	Log_AddMsg(3, 10, "Precharge state: Complete");
	Log_AddMsg(4, 30, "Precharge state: Fault");
	Log_AddMsg(5, 10, "Converter state: Standby");
	Log_AddMsg(6, 10, "Converter state: Waiting on precharge");
	Log_AddMsg(7, 10, "Converter state: Operating");
	Log_AddMsg(8, 30, "Converter state: Fault");

	return SAFE;
}


/**
 * Main interrupt routine
 */
tUserSafe UserInterrupt(void)
{
	core_state = GetCoreState();

	/**
	 * Retrieve all the measurements
	 * - Analog front-end configuration on B-Box RCP: https://imperix.com/doc/help/analog-front-end-configuration-on-b-box-rcp
	*/
	User_readADC();

	/**
	 * MPPT algorithm for PV panel
	 * - TN117 MPPT algorithms: https://imperix.com/doc/implementation/maximum-power-point-tracking-mppt
	*/
	User_runMPPT();

	/**
	 * PV current control with PI controller
	 * - TN105 PI controller implementation for current control: https://imperix.com/doc/implementation/pi-controller
	*/
	User_runPVCurrentCtrl();

	/**
	 * Park transform and DQPLL for grid angle reference
	 * - TN103 SRF PLL: https://imperix.com/doc/implementation/synchronous-reference-frame-pll
	*/
	User_runDQPLL();

	/**
	 * DC Bus voltage control (Cascaded voltage control over grid current vector control)
	 * - TN106 Vector current control: https://imperix.com/doc/implementation/vector-current-control
	 * - TN108 Cascaded voltage control: https://imperix.com/doc/implementation/cascaded-voltage-control
	*/
	User_runDCBusVoltageCtrl();

	/**
	 * Set the duty cycles for the CB_PWM modulators
	*/
	User_setDutyCycles();

	/**
	 * Tests the error conditions for the precharge statemachine
	 */
	User_TestErrors();

	/**
	 * State machines to automatically precharge and operate the converter
	 */
	User_RunPrechargeFSM();
	User_RunOperationFSM();

	/**
	 * Operates the relays according to the state machines states
	 * - TN 131 DC bus pre-charging techniques: https://imperix.com/doc/implementation/dc-bus-pre-charging-techniques
	 */
	User_operateRelays();

	/**
	 * Throw user fault
	 */
	User_throwUserFault();

	return SAFE;
}

// Background loop function for MPPT
tUserSafe UserBackground()
{
  if(SubTaskFlag)
  {
	if (activate_boost == 1 && core_state == 2)
		// run MPPT if boost is active and PWM enabled
		Ipv_ref = RunMPPTracking(&mpptracker, Ipv, Vpv*Ipv);
	else
	{
		// re-initialize the MPPT if the boost is deactivated or PWM disabled
		(&mpptracker)->meas_prev = 0;
		(&mpptracker)->power_prev = 0;
		(&mpptracker)->reference = 0;
		Ipv_ref = 0;
	}
    SubTaskFlag = false;
  }

  return SAFE;
}


void User_readADC()
{
	// Read all the ADC measured values
	Ig_abc.A = Adc_GetValue(ADC0);
	Ig_abc.B = Adc_GetValue(ADC1);
	Ig_abc.C = Adc_GetValue(ADC2);

	Ipv = -Adc_GetValue(ADC3);

	Vdc = Adc_GetValue(ADC8);

	Vpv = Adc_GetValue(ADC12);

	Vg_abc.A = Adc_GetValue(ADC13);
	Vg_abc.B = Adc_GetValue(ADC14);
	Vg_abc.C = Adc_GetValue(ADC15);

	// Assign values to global float variables to see them in Cockpit
	Vg_a = Vg_abc.A;
	Vg_b = Vg_abc.B;
	Vg_c = Vg_abc.C;

	Ig_a = Ig_abc.A;
	Ig_b = Ig_abc.B;
	Ig_c = Ig_abc.C;
}

void User_runMPPT()
{
	// Increment timer and trigger flag for background loop
	SubTaskTimer += SAMPLING_PERIOD;
	if(SubTaskTimer >= MPPT_PERIOD){
		SubTaskTimer = SubTaskTimer - MPPT_PERIOD;
		SubTaskFlag  = true;
	}
}

void User_runPVCurrentCtrl()
{
	// Execute the current controllers on the MPPT strings:
	Eb = Vpv - RunPIController(&Ipv_reg, Ipv_ref - Ipv);
}

void User_runDQPLL()
{
	abc2DQ0(&Vg_dq0,&Vg_abc,Theta);
	Theta = RunDQPLL(&DQPLL, &Vg_dq0);
	Vg_d = Vg_dq0.real;
	Vg_q = Vg_dq0.imaginary;
	w_grid = (&DQPLL)->omega;
}

void User_runDCBusVoltageCtrl()
{
	// Compute current feed-forward for grid current controller
	if (Vg_dq0.real > 0.01)
		Ig_d_feedforward = (1-K_IIR_LPF)*(2.0/3.0*(Ipv_ref * Vpv) / Vg_dq0.real) + K_IIR_LPF*Ig_d_feedforward;
	else
		Ig_d_feedforward = (1-K_IIR_LPF)*(2.0/3.0*(Ipv_ref * Vpv) / 0.01) + K_IIR_LPF*Ig_d_feedforward;

	// Run PI controller for grid current control
	Ig_dq0_ref.real = Ig_d_feedforward - RunPIController(&Vdc_reg, Vdc_ref-Vdc);
	Ig_dq0_ref.imaginary = Ig_q_ref;
	Ig_d_ref = Ig_dq0_ref.real; // monitoring in Cockpit

	// Convert grid current from time domain to synchronous reference frame
	abc2DQ0(&Ig_dq0, &Ig_abc, Theta);
	Ig_d = Ig_dq0.real;
	Ig_q = Ig_dq0.imaginary;

	// Vector current control
	Eg_dq0.real = Vg_dq0.real + RunPIController(&Ig_d_reg, Ig_dq0_ref.real-Ig_dq0.real) - OMEGA*LG*Ig_dq0.imaginary;
	Eg_dq0.imaginary = Vg_dq0.imaginary + RunPIController(&Ig_q_reg, Ig_dq0_ref.imaginary-Ig_dq0.imaginary) + OMEGA*LG*Ig_dq0.real;
	Eg_dq0.offset = 0;

	// Transform the converter voltage back in time domain
	DQ02abc(&Eg_abc, &Eg_dq0, Theta);
}

void User_setDutyCycles()
{
	if(activate_inverter)
	{
		CbPwm_Activate(PWM_CHANNEL_0);
		CbPwm_Activate(PWM_CHANNEL_1);
		CbPwm_Activate(PWM_CHANNEL_2);
	}
	else
	{
		CbPwm_Deactivate(PWM_CHANNEL_0);
		CbPwm_Deactivate(PWM_CHANNEL_1);
		CbPwm_Deactivate(PWM_CHANNEL_2);
	}

	if(activate_boost)
			CbPwm_Activate(PWM_CHANNEL_3);
		else CbPwm_Deactivate(PWM_CHANNEL_3);

	// Refresh the PWM duty-cycle of PWM channels related to the PV strings:
	CbPwm_SetDutyCycle(PWM_CHANNEL_0, 0.5 + Eg_abc.A/Vdc);
	CbPwm_SetDutyCycle(PWM_CHANNEL_1, 0.5 + Eg_abc.B/Vdc);
	CbPwm_SetDutyCycle(PWM_CHANNEL_2, 0.5 + Eg_abc.C/Vdc);
	CbPwm_SetDutyCycle(PWM_CHANNEL_3, Eb/Vdc);
}

void User_TestErrors()
{
	// PV voltage errors
	err_Vpv_low  = (Vpv < Vpv_min);
	err_Vpv_high = (Vpv > Vpv_max);

	// Grid frequency error
	err_No_GridSync = (w_grid < w_min || w_grid > w_max);

	// Vg_rectified = max of abs(phase-to-phase voltage)
	Vg_rectified = fabs(Vg_abc.A-Vg_abc.B);
	if (Vg_rectified < fabs(Vg_abc.B-Vg_abc.C))
		Vg_rectified = fabs(Vg_abc.B-Vg_abc.C);
	if (Vg_rectified < fabs(Vg_abc.C-Vg_abc.A))
		Vg_rectified = fabs(Vg_abc.C-Vg_abc.A);

	// Grid voltage error
	err_No_Grid = (Vg_rectified < Vgrid_min);

	// DC-link voltage too low error: the DC-link must be high enough for ending precharge, but its minimum value depends on the rectified grid voltage
	// due to dynamics of rectified grid voltage, hysteresis comparison is needed
	if (Vdc < 0.85*Vg_rectified)
		err_Vdc_low = true;
	else if (Vdc > 1.1*Vg_rectified)
		err_Vdc_low = false;
	// else: within hysteresis, keep err_Vdc_high unchanged

	// DC-link voltage too high error
	err_Vdc_high = (Vdc > Vdc_max);
}

void User_RunPrechargeFSM()
{
	/**
	 * This state machine controls the precharge of the DC bus.
	 * The user can switch the input "activate" between 1 and 0 to respectively start/stop the precharge.
	 * The state machine waits at least 250ms between the "CHARGING" and "CLOSING_BYPASS_RELAY" states.
	 * It also waits 200ms between the "CLOSING_BYPASS_RELAY" and "READY_TO_OPERATE" states, because the bypass relay needs a few millisecond to close.
	 * The output "Ready" indicates that the bus is precharged and the converter is ready to operate.
	 * If a fault occurs (CoreState is 0), the state machine goes in "DISCHARGING" state and opens all the relays for security.
	 * In simulation, the state machine goes in "SIMULATION" state, to avoid the simulation of the whole precharge. The user still can set the variable "SimulationMode" to 0 in order to simulate the precharge.
	*/
	switch (Next_state_precharge){
	// Init
	case PRECHARGE_INIT:
		State_precharge = PRECHARGE_INIT;
		Next_state_precharge = PRECHARGE_STANDBY;
		break;

	// The converter is disconnected from the grid and is in standby, all the relays are open
	case PRECHARGE_STANDBY:
		if (State_precharge != Next_state_precharge)
			Log_SendMsg(0, NULL, 0);
		State_precharge = Next_state_precharge;

		precharge_relay = 0;
		bypass_relay = 0;
		Precharge_ready = false;
		Precharge_fault = false;
		if (State_precharge != Next_state_precharge)
			precharge_cnt = 0;
		else
			precharge_cnt++;

		if (activate == 1 && core_state > 0 && precharge_cnt > 0.2*SW_FREQ)
		{
			Next_state_precharge = SYNCHRONIZING;
		}

		break;

	// The converter is synchronizing to the grid
	case SYNCHRONIZING:
		if (State_precharge != Next_state_precharge)
		{
			Log_SendMsg(1, NULL, 0);
			precharge_cnt = 0;
		}
		else
		{
			precharge_cnt++;
		}
		State_precharge = Next_state_precharge;

		Precharge_ready = false;
		Precharge_fault = false;
		precharge_relay = 0;
		bypass_relay = 0;


		if (precharge_cnt > 0.2*SW_FREQ && !err_No_GridSync && !err_No_Grid && core_state != 0)
		{
			Next_state_precharge = CHARGING;
		}
		else if(activate == 0 || core_state == 0)
		{
			Next_state_precharge = PRECHARGE_STANDBY;
		}
		else if(err_No_GridSync || err_No_Grid)
		{
			Next_state_precharge = PRECHARGE_FAULT;
		}

		break;

	// The DC bus of the converter is being charged from the grid (precharge relay closed)
	case CHARGING:
		if (State_precharge != Next_state_precharge)
		{
			Log_SendMsg(2, NULL, 0);
			precharge_cnt = 0;
		}
		else
		{
			precharge_cnt++;
		}
		State_precharge = Next_state_precharge;

		Precharge_ready = false;
		Precharge_fault = false;
		precharge_relay = 1;
		bypass_relay = 0;

		if (!err_Vpv_low && !err_Vpv_high && !err_Vdc_low && !err_Vdc_high)
		{
			Next_state_precharge = PRECHARGE_COMPLETE;
		}
		else if ((precharge_cnt > 3*SW_FREQ && (err_Vpv_low || err_Vpv_high || err_Vdc_low || err_Vdc_high)) || (precharge_cnt > 0.02*SW_FREQ && (err_No_GridSync || err_No_Grid)) || core_state == 0)
		{
			Next_state_precharge = PRECHARGE_FAULT;
		}
		else if (activate == 0)
		{
			Next_state_precharge = PRECHARGE_STANDBY;
		}

		break;

	// The DC bus is precharged and the converter is fully connected to the grid (bypass relay closed)
	case PRECHARGE_COMPLETE:
		if (State_precharge != Next_state_precharge)
			Log_SendMsg(3, NULL, 0);
		State_precharge = Next_state_precharge;

		precharge_relay = 1;
		bypass_relay = 1;
		Precharge_ready = true;
		Precharge_fault = false;

		if (err_Vpv_low || err_Vpv_high || err_Vdc_low || err_Vdc_high || (precharge_cnt > 0.02*SW_FREQ && (err_No_GridSync || err_No_Grid)) || core_state == 0)
		{
			Next_state_precharge = PRECHARGE_FAULT;
		}
		else if(activate == 0)
		{
			Next_state_precharge = PRECHARGE_STANDBY;
		}

		break;

	// A fault occurred during the precharge procedure, all the relays are open
	case PRECHARGE_FAULT:
		if (State_precharge != Next_state_precharge)
			Log_SendMsg(4, NULL, 0);
		State_precharge = Next_state_precharge;

		precharge_relay = 0;
		bypass_relay = 0;
		Precharge_ready = false;
		Precharge_fault = true;

		if(activate == 0)
		{
			Next_state_precharge = PRECHARGE_STANDBY;
		}

		break;
	}

	state_precharge_uint = (unsigned int) State_precharge;
}

void User_RunOperationFSM()
{
	switch (Next_state_operation){
	// Init
	case OP_INIT:
		State_operation = OP_INIT;
		Next_state_operation = OP_STANDBY;
		break;
	// The converter is in standby and is waiting to be turned on, everything is disabled
	case OP_STANDBY:
		if (State_operation != Next_state_operation)
					Log_SendMsg(5, NULL, 0);
		State_operation = Next_state_operation;

		activate_boost = 0;
		activate_inverter = 0;

		if (activate == 1 && core_state > 0)
		{
			Next_state_operation = OP_WAITING_ON_PRECHARGE;
		}

		break;

	// The converter is started and is waiting for the precharge procedure to complete
	case OP_WAITING_ON_PRECHARGE:
		if (State_operation != Next_state_operation)
		{
			Log_SendMsg(6, NULL, 0);
			operation_cnt = 0;
		}
		else
		{
			operation_cnt++;
		}
		State_operation = Next_state_operation;

		activate_boost = 0;
		activate_inverter = 0;

		if (core_state > 0 && Precharge_ready && operation_cnt > 1*SW_FREQ)
		{
			Next_state_operation = OP_READY_TO_OPERATE;
		}
		else if (activate == 0)
		{
			Next_state_operation = OP_STANDBY;
		}
		else if (Precharge_fault || core_state == 0 || operation_cnt > 30*SW_FREQ)
		{
			Next_state_operation = OP_FAULT;
		}

		break;

	// The converter is fully turned on, the PV relay is closed, the boost converter, the inverter and the PWM signals are enabled
	case OP_READY_TO_OPERATE:
		if (State_operation != Next_state_operation)
			Log_SendMsg(7, NULL, 0);
		State_operation = Next_state_operation;

		activate_boost = 1;
		activate_inverter = 1;

		if (activate == 0)
		{
			Next_state_operation = OP_STANDBY;
		}
		else if (Precharge_fault || core_state == 0)
		{
			Next_state_operation = OP_FAULT;
		}

		break;

	// A precharge or core fault was detected
	case OP_FAULT:
		if (State_operation != Next_state_operation)
			Log_SendMsg(8, NULL, 0);
		State_operation = Next_state_operation;

		activate_boost = 0;
		activate_inverter = 0;

		if (Precharge_fault == 0 && core_state > 0)
		{
			Next_state_operation = OP_STANDBY;
		}

		break;
	}

	state_operation_uint = (unsigned int) State_operation;
}

void User_operateRelays()
{
	// Operates the relay through the GPO of the B-Box RCP according to the current state of the converter
	if(precharge_relay) Gpo_SetBit(0, 0);
	else Gpo_ClearBit(0, 0);

	if(bypass_relay) Gpo_SetBit(1, 0);
	else Gpo_ClearBit(1, 0);
}

void User_throwUserFault()
{
	if (err_No_GridSync && Precharge_fault)
		SetUserFault("Precharge fault: Could not synchronize to grid");
	else if (err_No_Grid && Precharge_fault)
		SetUserFault("Precharge fault: No grid detected");
	else if (err_Vdc_high && Precharge_fault)
		SetUserFault("Precharge fault: High DC bus voltage");
	else if (err_Vdc_low && Precharge_fault)
		SetUserFault("Precharge fault: Low DC bus voltage");
	else if (err_Vpv_low && Precharge_fault)
		SetUserFault("Precharge fault: Low panel voltage");
	else if (err_Vpv_high && Precharge_fault)
		SetUserFault("Precharge fault: High panel voltage");

	if (!Precharge_fault)
		RecoverFromUserFault();
}

/**
 * Routine executed when the core state goes into FAULT mode
 */
void UserError(tErrorSource source)
{
	// Open relays
	//precharge_relay = 0;
	//bypass_relay = 0;
	//User_operateRelays();
}
