
// Central PV inverter (three-phase grid-tied) v2
// https://imperix.com/doc/example/three-phase-pv-inverter

#ifndef MY_FUNCTIONS_USER_H_
#define MY_FUNCTIONS_USER_H_

#include <math.h>
#include <stdlib.h>
#include "extern_user.h"

#include "sensors.h"

#include "Core/core.h"
#include "Core/interrupts.h"
#include "Driver/peripherals.h"

#include "../API/controllers.h"
#include "../API/transformations.h"
#include "../API/PLLs.h"


/**
 * Readability helpers
 */
#define ADC0 0
#define ADC1 1
#define ADC2 2
#define ADC3 3
#define ADC8 8
#define ADC12 12
#define ADC13 13
#define ADC14 14
#define ADC15 15

#define PI 3.1415926535

#define PHASE0 0										// Phase for the PWM modulators

#define OMEGA 2*PI*50
#define K_IIR_LPF 0.98									// Low pass filter coefficient

/**
 * Interrupt parameters
 */
#define SW_FREQ         (16e3)      					// Switching frequency (and main interrupt frequency)
#define SAMPLING_PERIOD (1/SW_FREQ) 					// Sampling period
#define MPPT_PERIOD 	0.005           				// MPPT period

#define DEADTIME 700e-9

/**
 * Application-specific parameters
 */
#define F_G 	(50)									// Grid frequency [Hz]
#define W_G 	(2*PI*F_G)								// Grid pulsation [rad/s]
#define LG  	(2.5e-3)								// Grid inductance [H]
#define RG  	(22e-3)									// ESR of grid inductance [Ohm]
#define CDC 	(4*260e-6)								// DC bus capacitance [F]
#define LB  	(2.5e-3)								// Boost inductance [H]
#define RB  	(22e-3)									// ESR of boost inductance [Ohm]
#define IS 		(40)                  					// PV panel's short-circuit current [A]
#define Voc		(400)                  					// PV panel's open-circuit voltage [V]

/**
 * Boost current controller parameters
 */
#define K1  (1/RB)
#define T1  (LB/RB)
#define Td1 ((1/2.0 + 1/2.0 + 2/3.0)*SAMPLING_PERIOD) // 1/2Ts of sampling, 1/2Ts of computation, 2/3 for the triangle carrier
#define Tn1 (T1)
#define Ti1 (2*K1*Td1)

/**
 * Grid currents controller parameters
 */
#define K2  (1/RG)
#define T2  (LG/RG)
#define Td2 ((1/2.0 + 1/2.0 + 2/3.0)*SAMPLING_PERIOD) // 1/2Ts of sampling, 1/2Ts of computation, 2/3 for the triangle carrier
#define Tn2 (T2)
#define Ti2 (2*K2*Td2)

/**
 * Enum for the precharge state machine states
 */
typedef enum {
	PRECHARGE_STANDBY 		= 0,
	SYNCHRONIZING  			= 1,
	CHARGING  				= 2,
	PRECHARGE_COMPLETE 		= 3,
	PRECHARGE_FAULT			= 4,
	PRECHARGE_INIT			= -1,
} tStatePrecharge;

/**
 * Enum for the operation state machine states
 */
typedef enum {
	OP_STANDBY 				= 0,
	OP_WAITING_ON_PRECHARGE = 1,
	OP_READY_TO_OPERATE		= 2,
	OP_FAULT				= 3,
	OP_INIT					= -1,
} tStateOperation;

/**
 * Main interrupt routine.
 * @param	void
 * @return	tUserSafe 		SAFE if the routine is executed without error, UNSAFE otherwise
 */
tUserSafe UserInterrupt(void);

/**
 * Routine to fully configure a PWM channel with complementary outputs and a carrier-based modulator
 * Must be called in UserInit()
 * @param	output			the PWM channel or lane to address (from tPwmOutput list)
 * @param	clock			the clock to use as reference for generating the PWM signals
 * @param	carrier			the PWM carrier shape to use (from tPwmCarrier list)
 * @param	deadTime		the dead-time duration between the high and low PWM signals in seconds
 * @param 	device			the id of the device to address (B-Box or B-Board) (optional, default = 0)
 * @return	void
 */
void CbPwm_ConfigureChannel(tPwmOutput output, tClock clock, tPwmCarrier carrier, float deadTime, float phase, unsigned int device=0){
	CbPwm_ConfigureClock(output, clock, device);
	CbPwm_ConfigureOutputMode(output, COMPLEMENTARY, device);
	CbPwm_ConfigureCarrier(output, carrier, device);
	CbPwm_ConfigureDeadTime(output, deadTime, device);
	CbPwm_SetDutyCycle(output, 0, device);
	CbPwm_SetPhase(output, phase, device);
}

#endif /* MY_FUNCTIONS_USER_H_ */
