
library IEEE;
use IEEE.STD_LOGIC_1164.ALL;
use IEEE.NUMERIC_STD.ALL;

-- ------------------------------------------------------------
-- IMPORTANTE NOTICE
-- ------------------------------------------------------------
-- Behavior
-- --------
-- * The number of in ports determines the maximal number of
--   signals that can be transmitted.
--   Exceeding signals are ignored.
-- * The number of out ports determines the maximal number of
--   signals that can be received.
--   Exceeding signals are ignored.
--
-- Number of ports
-- ---------------
-- * The number of in and out ports can be different.
-- * To extend the number of in and/or out ports of this module,
--   search (Ctrl+F) for 'flag' and apply the following changes:
--    - [flag  A   ] change the generic constant(s) ;
--    - [flags B, C] create the new ports ;
--    - [flags D, E] add the ports to the interface(s) ;
--    - [flags F, G] extend the related logic.
--
-- Provided files and project
-- --------------------------
-- * This project was tested with 32 signals exchanged
--   in each direction.
-- * This project was tested with an OPAL-RT OP4510.
-- ------------------------------------------------------------

entity aurora_ix_opalrt_driver is
	Generic(

		-- SBIO-related constants
		SBIO_ADDR_WIDTH : natural := 10;
		SBIO_DATA_WIDTH : natural := 16;

		-- Number of ports
		NUM_PORTS_TX : natural range 1 to 500 := 50;
		NUM_PORTS_RX : natural range 1 to 500 := 50

		-- [flag A]

	);
	Port(

		-- main clock at 250MHz
		clk_250_mhz : in std_logic;

		-- main reset
		reset : in std_logic;

		-- transmission trigger
		-- (active-high, ignored if already transmitting)
		tx_trigger : in std_logic;

		-- outgoing signals to be transmitted to the RT-Box
		tx_signal_00 : in std_logic_vector(31 downto 0);
		tx_signal_01 : in std_logic_vector(31 downto 0);
		tx_signal_02 : in std_logic_vector(31 downto 0);
		tx_signal_03 : in std_logic_vector(31 downto 0);
		tx_signal_04 : in std_logic_vector(31 downto 0);
		tx_signal_05 : in std_logic_vector(31 downto 0);
		tx_signal_06 : in std_logic_vector(31 downto 0);
		tx_signal_07 : in std_logic_vector(31 downto 0);
		tx_signal_08 : in std_logic_vector(31 downto 0);
		tx_signal_09 : in std_logic_vector(31 downto 0);
		tx_signal_10 : in std_logic_vector(31 downto 0);
		tx_signal_11 : in std_logic_vector(31 downto 0);
		tx_signal_12 : in std_logic_vector(31 downto 0);
		tx_signal_13 : in std_logic_vector(31 downto 0);
		tx_signal_14 : in std_logic_vector(31 downto 0);
		tx_signal_15 : in std_logic_vector(31 downto 0);
		tx_signal_16 : in std_logic_vector(31 downto 0);
		tx_signal_17 : in std_logic_vector(31 downto 0);
		tx_signal_18 : in std_logic_vector(31 downto 0);
		tx_signal_19 : in std_logic_vector(31 downto 0);
		tx_signal_20 : in std_logic_vector(31 downto 0);
		tx_signal_21 : in std_logic_vector(31 downto 0);
		tx_signal_22 : in std_logic_vector(31 downto 0);
		tx_signal_23 : in std_logic_vector(31 downto 0);
		tx_signal_24 : in std_logic_vector(31 downto 0);
		tx_signal_25 : in std_logic_vector(31 downto 0);
		tx_signal_26 : in std_logic_vector(31 downto 0);
		tx_signal_27 : in std_logic_vector(31 downto 0);
		tx_signal_28 : in std_logic_vector(31 downto 0);
		tx_signal_29 : in std_logic_vector(31 downto 0);
		tx_signal_30 : in std_logic_vector(31 downto 0);
		tx_signal_31 : in std_logic_vector(31 downto 0);
		tx_signal_32 : in std_logic_vector(31 downto 0);
		tx_signal_33 : in std_logic_vector(31 downto 0);
		tx_signal_34 : in std_logic_vector(31 downto 0);
		tx_signal_35 : in std_logic_vector(31 downto 0);
		tx_signal_36 : in std_logic_vector(31 downto 0);
		tx_signal_37 : in std_logic_vector(31 downto 0);
		tx_signal_38 : in std_logic_vector(31 downto 0);
		tx_signal_39 : in std_logic_vector(31 downto 0);
		tx_signal_40 : in std_logic_vector(31 downto 0);
		tx_signal_41 : in std_logic_vector(31 downto 0);
		tx_signal_42 : in std_logic_vector(31 downto 0);
		tx_signal_43 : in std_logic_vector(31 downto 0);
		tx_signal_44 : in std_logic_vector(31 downto 0);
		tx_signal_45 : in std_logic_vector(31 downto 0);
		tx_signal_46 : in std_logic_vector(31 downto 0);
		tx_signal_47 : in std_logic_vector(31 downto 0);
		tx_signal_48 : in std_logic_vector(31 downto 0);
		tx_signal_49 : in std_logic_vector(31 downto 0);

		-- [flag B]

		-- number of signals to send to the RT-Box
		tx_n_signals : in unsigned(15 downto 0);

		-- incoming signals received from the RT-Box
		rx_signal_00 : out std_logic_vector(31 downto 0);
		rx_signal_01 : out std_logic_vector(31 downto 0);
		rx_signal_02 : out std_logic_vector(31 downto 0);
		rx_signal_03 : out std_logic_vector(31 downto 0);
		rx_signal_04 : out std_logic_vector(31 downto 0);
		rx_signal_05 : out std_logic_vector(31 downto 0);
		rx_signal_06 : out std_logic_vector(31 downto 0);
		rx_signal_07 : out std_logic_vector(31 downto 0);
		rx_signal_08 : out std_logic_vector(31 downto 0);
		rx_signal_09 : out std_logic_vector(31 downto 0);
		rx_signal_10 : out std_logic_vector(31 downto 0);
		rx_signal_11 : out std_logic_vector(31 downto 0);
		rx_signal_12 : out std_logic_vector(31 downto 0);
		rx_signal_13 : out std_logic_vector(31 downto 0);
		rx_signal_14 : out std_logic_vector(31 downto 0);
		rx_signal_15 : out std_logic_vector(31 downto 0);
		rx_signal_16 : out std_logic_vector(31 downto 0);
		rx_signal_17 : out std_logic_vector(31 downto 0);
		rx_signal_18 : out std_logic_vector(31 downto 0);
		rx_signal_19 : out std_logic_vector(31 downto 0);
		rx_signal_20 : out std_logic_vector(31 downto 0);
		rx_signal_21 : out std_logic_vector(31 downto 0);
		rx_signal_22 : out std_logic_vector(31 downto 0);
		rx_signal_23 : out std_logic_vector(31 downto 0);
		rx_signal_24 : out std_logic_vector(31 downto 0);
		rx_signal_25 : out std_logic_vector(31 downto 0);
		rx_signal_26 : out std_logic_vector(31 downto 0);
		rx_signal_27 : out std_logic_vector(31 downto 0);
		rx_signal_28 : out std_logic_vector(31 downto 0);
		rx_signal_29 : out std_logic_vector(31 downto 0);
		rx_signal_30 : out std_logic_vector(31 downto 0);
		rx_signal_31 : out std_logic_vector(31 downto 0);
		rx_signal_32 : out std_logic_vector(31 downto 0);
		rx_signal_33 : out std_logic_vector(31 downto 0);
		rx_signal_34 : out std_logic_vector(31 downto 0);
		rx_signal_35 : out std_logic_vector(31 downto 0);
		rx_signal_36 : out std_logic_vector(31 downto 0);
		rx_signal_37 : out std_logic_vector(31 downto 0);
		rx_signal_38 : out std_logic_vector(31 downto 0);
		rx_signal_39 : out std_logic_vector(31 downto 0);
		rx_signal_40 : out std_logic_vector(31 downto 0);
		rx_signal_41 : out std_logic_vector(31 downto 0);
		rx_signal_42 : out std_logic_vector(31 downto 0);
		rx_signal_43 : out std_logic_vector(31 downto 0);
		rx_signal_44 : out std_logic_vector(31 downto 0);
		rx_signal_45 : out std_logic_vector(31 downto 0);
		rx_signal_46 : out std_logic_vector(31 downto 0);
		rx_signal_47 : out std_logic_vector(31 downto 0);
		rx_signal_48 : out std_logic_vector(31 downto 0);
		rx_signal_49 : out std_logic_vector(31 downto 0);

		-- [flag C]

		-- valid signal indicating that RX signals were updated
		rx_vld : out std_logic := '0';

		-- information about the RX signals
		rx_n_signals : out unsigned(15 downto 0) := (others => '0');

		-- outgoing FIFO interface
		M_AXIS_TX_tdata : out std_logic_vector(31 downto 0);
		M_AXIS_TX_tvalid : out std_logic;
		M_AXIS_TX_tlast : out std_logic;
		M_AXIS_TX_tready : in std_logic;
		M_AXIS_TX_tuser : out std_logic_vector(3 downto 0);

		-- incoming FIFO interface
		S_AXIS_RX_tdata : in std_logic_vector(31 downto 0);
		S_AXIS_RX_tvalid : in std_logic;
		S_AXIS_RX_tlast : in std_logic;
		S_AXIS_RX_tready : out std_logic;
		S_AXIS_RX_tuser : in std_logic_vector(3 downto 0)

	);
end aurora_ix_opalrt_driver;

architecture Behavioral of aurora_ix_opalrt_driver is

	-----------------------------------------------
	-- INTERFACES
	-----------------------------------------------

	ATTRIBUTE X_INTERFACE_INFO : STRING;
	ATTRIBUTE X_INTERFACE_PARAMETER : STRING;

	ATTRIBUTE X_INTERFACE_INFO of M_AXIS_TX_tdata  : SIGNAL is "xilinx.com:interface:axis:1.0 M_AXIS_TX tdata";
	ATTRIBUTE X_INTERFACE_INFO of M_AXIS_TX_tvalid : SIGNAL is "xilinx.com:interface:axis:1.0 M_AXIS_TX tvalid";
	ATTRIBUTE X_INTERFACE_INFO of M_AXIS_TX_tlast  : SIGNAL is "xilinx.com:interface:axis:1.0 M_AXIS_TX tlast";
	ATTRIBUTE X_INTERFACE_INFO of M_AXIS_TX_tready : SIGNAL is "xilinx.com:interface:axis:1.0 M_AXIS_TX tready";
	ATTRIBUTE X_INTERFACE_INFO of M_AXIS_TX_tuser  : SIGNAL is "xilinx.com:interface:axis:1.0 M_AXIS_TX tuser";

	ATTRIBUTE X_INTERFACE_INFO of S_AXIS_RX_tdata  : SIGNAL is "xilinx.com:interface:axis:1.0 S_AXIS_RX tdata";
	ATTRIBUTE X_INTERFACE_INFO of S_AXIS_RX_tvalid : SIGNAL is "xilinx.com:interface:axis:1.0 S_AXIS_RX tvalid";
	ATTRIBUTE X_INTERFACE_INFO of S_AXIS_RX_tlast  : SIGNAL is "xilinx.com:interface:axis:1.0 S_AXIS_RX tlast";
	ATTRIBUTE X_INTERFACE_INFO of S_AXIS_RX_tready : SIGNAL is "xilinx.com:interface:axis:1.0 S_AXIS_RX tready";
	ATTRIBUTE X_INTERFACE_INFO of S_AXIS_RX_tuser  : SIGNAL is "xilinx.com:interface:axis:1.0 S_AXIS_RX tuser";

	ATTRIBUTE X_INTERFACE_INFO of tx_signal_00 : SIGNAL is "imperix.ch:ix:user_regs_rtl:1.0 TX_SIGNALS reg_00";
	ATTRIBUTE X_INTERFACE_INFO of tx_signal_01 : SIGNAL is "imperix.ch:ix:user_regs_rtl:1.0 TX_SIGNALS reg_01";
	ATTRIBUTE X_INTERFACE_INFO of tx_signal_02 : SIGNAL is "imperix.ch:ix:user_regs_rtl:1.0 TX_SIGNALS reg_02";
	ATTRIBUTE X_INTERFACE_INFO of tx_signal_03 : SIGNAL is "imperix.ch:ix:user_regs_rtl:1.0 TX_SIGNALS reg_03";
	ATTRIBUTE X_INTERFACE_INFO of tx_signal_04 : SIGNAL is "imperix.ch:ix:user_regs_rtl:1.0 TX_SIGNALS reg_04";
	ATTRIBUTE X_INTERFACE_INFO of tx_signal_05 : SIGNAL is "imperix.ch:ix:user_regs_rtl:1.0 TX_SIGNALS reg_05";
	ATTRIBUTE X_INTERFACE_INFO of tx_signal_06 : SIGNAL is "imperix.ch:ix:user_regs_rtl:1.0 TX_SIGNALS reg_06";
	ATTRIBUTE X_INTERFACE_INFO of tx_signal_07 : SIGNAL is "imperix.ch:ix:user_regs_rtl:1.0 TX_SIGNALS reg_07";
	ATTRIBUTE X_INTERFACE_INFO of tx_signal_08 : SIGNAL is "imperix.ch:ix:user_regs_rtl:1.0 TX_SIGNALS reg_08";
	ATTRIBUTE X_INTERFACE_INFO of tx_signal_09 : SIGNAL is "imperix.ch:ix:user_regs_rtl:1.0 TX_SIGNALS reg_09";
	ATTRIBUTE X_INTERFACE_INFO of tx_signal_10 : SIGNAL is "imperix.ch:ix:user_regs_rtl:1.0 TX_SIGNALS reg_10";
	ATTRIBUTE X_INTERFACE_INFO of tx_signal_11 : SIGNAL is "imperix.ch:ix:user_regs_rtl:1.0 TX_SIGNALS reg_11";
	ATTRIBUTE X_INTERFACE_INFO of tx_signal_12 : SIGNAL is "imperix.ch:ix:user_regs_rtl:1.0 TX_SIGNALS reg_12";
	ATTRIBUTE X_INTERFACE_INFO of tx_signal_13 : SIGNAL is "imperix.ch:ix:user_regs_rtl:1.0 TX_SIGNALS reg_13";
	ATTRIBUTE X_INTERFACE_INFO of tx_signal_14 : SIGNAL is "imperix.ch:ix:user_regs_rtl:1.0 TX_SIGNALS reg_14";
	ATTRIBUTE X_INTERFACE_INFO of tx_signal_15 : SIGNAL is "imperix.ch:ix:user_regs_rtl:1.0 TX_SIGNALS reg_15";
	ATTRIBUTE X_INTERFACE_INFO of tx_signal_16 : SIGNAL is "imperix.ch:ix:user_regs_rtl:1.0 TX_SIGNALS reg_16";
	ATTRIBUTE X_INTERFACE_INFO of tx_signal_17 : SIGNAL is "imperix.ch:ix:user_regs_rtl:1.0 TX_SIGNALS reg_17";
	ATTRIBUTE X_INTERFACE_INFO of tx_signal_18 : SIGNAL is "imperix.ch:ix:user_regs_rtl:1.0 TX_SIGNALS reg_18";
	ATTRIBUTE X_INTERFACE_INFO of tx_signal_19 : SIGNAL is "imperix.ch:ix:user_regs_rtl:1.0 TX_SIGNALS reg_19";
	ATTRIBUTE X_INTERFACE_INFO of tx_signal_20 : SIGNAL is "imperix.ch:ix:user_regs_rtl:1.0 TX_SIGNALS reg_20";
	ATTRIBUTE X_INTERFACE_INFO of tx_signal_21 : SIGNAL is "imperix.ch:ix:user_regs_rtl:1.0 TX_SIGNALS reg_21";
	ATTRIBUTE X_INTERFACE_INFO of tx_signal_22 : SIGNAL is "imperix.ch:ix:user_regs_rtl:1.0 TX_SIGNALS reg_22";
	ATTRIBUTE X_INTERFACE_INFO of tx_signal_23 : SIGNAL is "imperix.ch:ix:user_regs_rtl:1.0 TX_SIGNALS reg_23";
	ATTRIBUTE X_INTERFACE_INFO of tx_signal_24 : SIGNAL is "imperix.ch:ix:user_regs_rtl:1.0 TX_SIGNALS reg_24";
	ATTRIBUTE X_INTERFACE_INFO of tx_signal_25 : SIGNAL is "imperix.ch:ix:user_regs_rtl:1.0 TX_SIGNALS reg_25";
	ATTRIBUTE X_INTERFACE_INFO of tx_signal_26 : SIGNAL is "imperix.ch:ix:user_regs_rtl:1.0 TX_SIGNALS reg_26";
	ATTRIBUTE X_INTERFACE_INFO of tx_signal_27 : SIGNAL is "imperix.ch:ix:user_regs_rtl:1.0 TX_SIGNALS reg_27";
	ATTRIBUTE X_INTERFACE_INFO of tx_signal_28 : SIGNAL is "imperix.ch:ix:user_regs_rtl:1.0 TX_SIGNALS reg_28";
	ATTRIBUTE X_INTERFACE_INFO of tx_signal_29 : SIGNAL is "imperix.ch:ix:user_regs_rtl:1.0 TX_SIGNALS reg_29";
	ATTRIBUTE X_INTERFACE_INFO of tx_signal_30 : SIGNAL is "imperix.ch:ix:user_regs_rtl:1.0 TX_SIGNALS reg_30";
	ATTRIBUTE X_INTERFACE_INFO of tx_signal_31 : SIGNAL is "imperix.ch:ix:user_regs_rtl:1.0 TX_SIGNALS reg_31";
	ATTRIBUTE X_INTERFACE_INFO of tx_signal_32 : SIGNAL is "imperix.ch:ix:user_regs_rtl:1.0 TX_SIGNALS reg_32";
	ATTRIBUTE X_INTERFACE_INFO of tx_signal_33 : SIGNAL is "imperix.ch:ix:user_regs_rtl:1.0 TX_SIGNALS reg_33";
	ATTRIBUTE X_INTERFACE_INFO of tx_signal_34 : SIGNAL is "imperix.ch:ix:user_regs_rtl:1.0 TX_SIGNALS reg_34";
	ATTRIBUTE X_INTERFACE_INFO of tx_signal_35 : SIGNAL is "imperix.ch:ix:user_regs_rtl:1.0 TX_SIGNALS reg_35";
	ATTRIBUTE X_INTERFACE_INFO of tx_signal_36 : SIGNAL is "imperix.ch:ix:user_regs_rtl:1.0 TX_SIGNALS reg_36";
	ATTRIBUTE X_INTERFACE_INFO of tx_signal_37 : SIGNAL is "imperix.ch:ix:user_regs_rtl:1.0 TX_SIGNALS reg_37";
	ATTRIBUTE X_INTERFACE_INFO of tx_signal_38 : SIGNAL is "imperix.ch:ix:user_regs_rtl:1.0 TX_SIGNALS reg_38";
	ATTRIBUTE X_INTERFACE_INFO of tx_signal_39 : SIGNAL is "imperix.ch:ix:user_regs_rtl:1.0 TX_SIGNALS reg_39";
	ATTRIBUTE X_INTERFACE_INFO of tx_signal_40 : SIGNAL is "imperix.ch:ix:user_regs_rtl:1.0 TX_SIGNALS reg_40";
	ATTRIBUTE X_INTERFACE_INFO of tx_signal_41 : SIGNAL is "imperix.ch:ix:user_regs_rtl:1.0 TX_SIGNALS reg_41";
	ATTRIBUTE X_INTERFACE_INFO of tx_signal_42 : SIGNAL is "imperix.ch:ix:user_regs_rtl:1.0 TX_SIGNALS reg_42";
	ATTRIBUTE X_INTERFACE_INFO of tx_signal_43 : SIGNAL is "imperix.ch:ix:user_regs_rtl:1.0 TX_SIGNALS reg_43";
	ATTRIBUTE X_INTERFACE_INFO of tx_signal_44 : SIGNAL is "imperix.ch:ix:user_regs_rtl:1.0 TX_SIGNALS reg_44";
	ATTRIBUTE X_INTERFACE_INFO of tx_signal_45 : SIGNAL is "imperix.ch:ix:user_regs_rtl:1.0 TX_SIGNALS reg_45";
	ATTRIBUTE X_INTERFACE_INFO of tx_signal_46 : SIGNAL is "imperix.ch:ix:user_regs_rtl:1.0 TX_SIGNALS reg_46";
	ATTRIBUTE X_INTERFACE_INFO of tx_signal_47 : SIGNAL is "imperix.ch:ix:user_regs_rtl:1.0 TX_SIGNALS reg_47";
	ATTRIBUTE X_INTERFACE_INFO of tx_signal_48 : SIGNAL is "imperix.ch:ix:user_regs_rtl:1.0 TX_SIGNALS reg_48";
	ATTRIBUTE X_INTERFACE_INFO of tx_signal_49 : SIGNAL is "imperix.ch:ix:user_regs_rtl:1.0 TX_SIGNALS reg_49";

	-- [flag D]

	ATTRIBUTE X_INTERFACE_INFO of rx_signal_00 : SIGNAL is "imperix.ch:ix:user_regs_rtl:1.0 RX_SIGNALS reg_00";
	ATTRIBUTE X_INTERFACE_INFO of rx_signal_01 : SIGNAL is "imperix.ch:ix:user_regs_rtl:1.0 RX_SIGNALS reg_01";
	ATTRIBUTE X_INTERFACE_INFO of rx_signal_02 : SIGNAL is "imperix.ch:ix:user_regs_rtl:1.0 RX_SIGNALS reg_02";
	ATTRIBUTE X_INTERFACE_INFO of rx_signal_03 : SIGNAL is "imperix.ch:ix:user_regs_rtl:1.0 RX_SIGNALS reg_03";
	ATTRIBUTE X_INTERFACE_INFO of rx_signal_04 : SIGNAL is "imperix.ch:ix:user_regs_rtl:1.0 RX_SIGNALS reg_04";
	ATTRIBUTE X_INTERFACE_INFO of rx_signal_05 : SIGNAL is "imperix.ch:ix:user_regs_rtl:1.0 RX_SIGNALS reg_05";
	ATTRIBUTE X_INTERFACE_INFO of rx_signal_06 : SIGNAL is "imperix.ch:ix:user_regs_rtl:1.0 RX_SIGNALS reg_06";
	ATTRIBUTE X_INTERFACE_INFO of rx_signal_07 : SIGNAL is "imperix.ch:ix:user_regs_rtl:1.0 RX_SIGNALS reg_07";
	ATTRIBUTE X_INTERFACE_INFO of rx_signal_08 : SIGNAL is "imperix.ch:ix:user_regs_rtl:1.0 RX_SIGNALS reg_08";
	ATTRIBUTE X_INTERFACE_INFO of rx_signal_09 : SIGNAL is "imperix.ch:ix:user_regs_rtl:1.0 RX_SIGNALS reg_09";
	ATTRIBUTE X_INTERFACE_INFO of rx_signal_10 : SIGNAL is "imperix.ch:ix:user_regs_rtl:1.0 RX_SIGNALS reg_10";
	ATTRIBUTE X_INTERFACE_INFO of rx_signal_11 : SIGNAL is "imperix.ch:ix:user_regs_rtl:1.0 RX_SIGNALS reg_11";
	ATTRIBUTE X_INTERFACE_INFO of rx_signal_12 : SIGNAL is "imperix.ch:ix:user_regs_rtl:1.0 RX_SIGNALS reg_12";
	ATTRIBUTE X_INTERFACE_INFO of rx_signal_13 : SIGNAL is "imperix.ch:ix:user_regs_rtl:1.0 RX_SIGNALS reg_13";
	ATTRIBUTE X_INTERFACE_INFO of rx_signal_14 : SIGNAL is "imperix.ch:ix:user_regs_rtl:1.0 RX_SIGNALS reg_14";
	ATTRIBUTE X_INTERFACE_INFO of rx_signal_15 : SIGNAL is "imperix.ch:ix:user_regs_rtl:1.0 RX_SIGNALS reg_15";
	ATTRIBUTE X_INTERFACE_INFO of rx_signal_16 : SIGNAL is "imperix.ch:ix:user_regs_rtl:1.0 RX_SIGNALS reg_16";
	ATTRIBUTE X_INTERFACE_INFO of rx_signal_17 : SIGNAL is "imperix.ch:ix:user_regs_rtl:1.0 RX_SIGNALS reg_17";
	ATTRIBUTE X_INTERFACE_INFO of rx_signal_18 : SIGNAL is "imperix.ch:ix:user_regs_rtl:1.0 RX_SIGNALS reg_18";
	ATTRIBUTE X_INTERFACE_INFO of rx_signal_19 : SIGNAL is "imperix.ch:ix:user_regs_rtl:1.0 RX_SIGNALS reg_19";
	ATTRIBUTE X_INTERFACE_INFO of rx_signal_20 : SIGNAL is "imperix.ch:ix:user_regs_rtl:1.0 RX_SIGNALS reg_20";
	ATTRIBUTE X_INTERFACE_INFO of rx_signal_21 : SIGNAL is "imperix.ch:ix:user_regs_rtl:1.0 RX_SIGNALS reg_21";
	ATTRIBUTE X_INTERFACE_INFO of rx_signal_22 : SIGNAL is "imperix.ch:ix:user_regs_rtl:1.0 RX_SIGNALS reg_22";
	ATTRIBUTE X_INTERFACE_INFO of rx_signal_23 : SIGNAL is "imperix.ch:ix:user_regs_rtl:1.0 RX_SIGNALS reg_23";
	ATTRIBUTE X_INTERFACE_INFO of rx_signal_24 : SIGNAL is "imperix.ch:ix:user_regs_rtl:1.0 RX_SIGNALS reg_24";
	ATTRIBUTE X_INTERFACE_INFO of rx_signal_25 : SIGNAL is "imperix.ch:ix:user_regs_rtl:1.0 RX_SIGNALS reg_25";
	ATTRIBUTE X_INTERFACE_INFO of rx_signal_26 : SIGNAL is "imperix.ch:ix:user_regs_rtl:1.0 RX_SIGNALS reg_26";
	ATTRIBUTE X_INTERFACE_INFO of rx_signal_27 : SIGNAL is "imperix.ch:ix:user_regs_rtl:1.0 RX_SIGNALS reg_27";
	ATTRIBUTE X_INTERFACE_INFO of rx_signal_28 : SIGNAL is "imperix.ch:ix:user_regs_rtl:1.0 RX_SIGNALS reg_28";
	ATTRIBUTE X_INTERFACE_INFO of rx_signal_29 : SIGNAL is "imperix.ch:ix:user_regs_rtl:1.0 RX_SIGNALS reg_29";
	ATTRIBUTE X_INTERFACE_INFO of rx_signal_30 : SIGNAL is "imperix.ch:ix:user_regs_rtl:1.0 RX_SIGNALS reg_30";
	ATTRIBUTE X_INTERFACE_INFO of rx_signal_31 : SIGNAL is "imperix.ch:ix:user_regs_rtl:1.0 RX_SIGNALS reg_31";
	ATTRIBUTE X_INTERFACE_INFO of rx_signal_32 : SIGNAL is "imperix.ch:ix:user_regs_rtl:1.0 RX_SIGNALS reg_32";
	ATTRIBUTE X_INTERFACE_INFO of rx_signal_33 : SIGNAL is "imperix.ch:ix:user_regs_rtl:1.0 RX_SIGNALS reg_33";
	ATTRIBUTE X_INTERFACE_INFO of rx_signal_34 : SIGNAL is "imperix.ch:ix:user_regs_rtl:1.0 RX_SIGNALS reg_34";
	ATTRIBUTE X_INTERFACE_INFO of rx_signal_35 : SIGNAL is "imperix.ch:ix:user_regs_rtl:1.0 RX_SIGNALS reg_35";
	ATTRIBUTE X_INTERFACE_INFO of rx_signal_36 : SIGNAL is "imperix.ch:ix:user_regs_rtl:1.0 RX_SIGNALS reg_36";
	ATTRIBUTE X_INTERFACE_INFO of rx_signal_37 : SIGNAL is "imperix.ch:ix:user_regs_rtl:1.0 RX_SIGNALS reg_37";
	ATTRIBUTE X_INTERFACE_INFO of rx_signal_38 : SIGNAL is "imperix.ch:ix:user_regs_rtl:1.0 RX_SIGNALS reg_38";
	ATTRIBUTE X_INTERFACE_INFO of rx_signal_39 : SIGNAL is "imperix.ch:ix:user_regs_rtl:1.0 RX_SIGNALS reg_39";
	ATTRIBUTE X_INTERFACE_INFO of rx_signal_40 : SIGNAL is "imperix.ch:ix:user_regs_rtl:1.0 RX_SIGNALS reg_40";
	ATTRIBUTE X_INTERFACE_INFO of rx_signal_41 : SIGNAL is "imperix.ch:ix:user_regs_rtl:1.0 RX_SIGNALS reg_41";
	ATTRIBUTE X_INTERFACE_INFO of rx_signal_42 : SIGNAL is "imperix.ch:ix:user_regs_rtl:1.0 RX_SIGNALS reg_42";
	ATTRIBUTE X_INTERFACE_INFO of rx_signal_43 : SIGNAL is "imperix.ch:ix:user_regs_rtl:1.0 RX_SIGNALS reg_43";
	ATTRIBUTE X_INTERFACE_INFO of rx_signal_44 : SIGNAL is "imperix.ch:ix:user_regs_rtl:1.0 RX_SIGNALS reg_44";
	ATTRIBUTE X_INTERFACE_INFO of rx_signal_45 : SIGNAL is "imperix.ch:ix:user_regs_rtl:1.0 RX_SIGNALS reg_45";
	ATTRIBUTE X_INTERFACE_INFO of rx_signal_46 : SIGNAL is "imperix.ch:ix:user_regs_rtl:1.0 RX_SIGNALS reg_46";
	ATTRIBUTE X_INTERFACE_INFO of rx_signal_47 : SIGNAL is "imperix.ch:ix:user_regs_rtl:1.0 RX_SIGNALS reg_47";
	ATTRIBUTE X_INTERFACE_INFO of rx_signal_48 : SIGNAL is "imperix.ch:ix:user_regs_rtl:1.0 RX_SIGNALS reg_48";
	ATTRIBUTE X_INTERFACE_INFO of rx_signal_49 : SIGNAL is "imperix.ch:ix:user_regs_rtl:1.0 RX_SIGNALS reg_49";

	-- [flag E]

	ATTRIBUTE X_INTERFACE_INFO of clk_250_mhz: SIGNAL is "xilinx.com:signal:clock:1.0 clk_250_mhz CLK";
	ATTRIBUTE X_INTERFACE_PARAMETER of clk_250_mhz: SIGNAL is "ASSOCIATED_BUSIF M_AXIS_TX:S_AXIS_RX:TX_SIGNALS:RX_SIGNALS";

	-----------------------------------------------
	-- HELPER FUNCTION
	-----------------------------------------------

	function ceil_div2(x : unsigned) return unsigned is
	begin
		return (x + 1) / 2;
	end function;

	-----------------------------------------------
	-- SIGNALS DEFINITION
	-----------------------------------------------

	constant NUM_PORTS_TX_IS_ODD : natural := NUM_PORTS_TX mod 2;
	constant MAX_WORDS_TX : natural := NUM_PORTS_TX;
	constant MAX_WORDS_RX : natural := NUM_PORTS_RX;

	-----------------------------------------------

	type signals_regs_t is array(integer range <>) of std_logic_vector(31 downto 0);

	signal tx_signals_arr : signals_regs_t(0 to MAX_WORDS_TX-1) := (others => (others => '0'));
	signal rx_signals_arr : signals_regs_t(0 to MAX_WORDS_RX-1) := (others => (others => '0'));

	-----------------------------------------------

	signal tx_n_signals_i : natural range 0 to NUM_PORTS_TX;
	signal tx_n_signals_i_is_odd : std_logic;
	signal tx_n_words_i : natural range 0 to MAX_WORDS_TX;

	type tx_state_t is (TX_IDLE, TX_TRANSMISSION);
	signal tx_state : tx_state_t := TX_IDLE;

	signal tx_word_index : natural range 0 to MAX_WORDS_TX-1;

	signal tx_in_transmission : std_logic;
	signal tx_last_word : std_logic;

	signal tx_valid_transmission : std_logic;
	signal tx_valid_last_transmission : std_logic;

	signal tx_M_AXIS_TX_tvalid : std_logic;
	signal tx_M_AXIS_TX_tlast : std_logic;

	-----------------------------------------------

	signal rx_valid_transmission : std_logic;
	signal rx_valid_last_transmission : std_logic;
	signal rx_valid_last_transmission_ff : std_logic;

	signal rx_word_index : natural range 0 to MAX_WORDS_RX;

	signal rx_word_index_exceeding : std_logic;

	signal r_S_AXIS_RX_tready : std_logic;

	-----------------------------------------------
	-- DEBUG ATTRIBUTE
	-----------------------------------------------

	attribute MARK_DEBUG : string;

	attribute MARK_DEBUG of tx_signals_arr                : signal is "TRUE";
	attribute MARK_DEBUG of rx_signals_arr                : signal is "TRUE";

	attribute MARK_DEBUG of tx_n_signals_i                : signal is "TRUE";
	attribute MARK_DEBUG of tx_n_signals_i_is_odd         : signal is "TRUE";
	attribute MARK_DEBUG of tx_n_words_i                  : signal is "TRUE";
	attribute MARK_DEBUG of tx_state                      : signal is "TRUE";
	attribute MARK_DEBUG of tx_word_index                 : signal is "TRUE";
	attribute MARK_DEBUG of tx_in_transmission            : signal is "TRUE";
	attribute MARK_DEBUG of tx_last_word                  : signal is "TRUE";
	attribute MARK_DEBUG of tx_valid_transmission         : signal is "TRUE";
	attribute MARK_DEBUG of tx_valid_last_transmission    : signal is "TRUE";
	attribute MARK_DEBUG of tx_M_AXIS_TX_tvalid           : signal is "TRUE";
	attribute MARK_DEBUG of tx_M_AXIS_TX_tlast            : signal is "TRUE";

	attribute MARK_DEBUG of rx_valid_transmission         : signal is "TRUE";
	attribute MARK_DEBUG of rx_valid_last_transmission    : signal is "TRUE";
	attribute MARK_DEBUG of rx_valid_last_transmission_ff : signal is "TRUE";
	attribute MARK_DEBUG of rx_word_index                 : signal is "TRUE";
	attribute MARK_DEBUG of rx_word_index_exceeding       : signal is "TRUE";

begin

	-----------------------------------------------
	-- TRANSMISSION LOGIC (TX)
	-----------------------------------------------

	-- Sample inputs on trigger (when idle)
	process(clk_250_mhz) begin
		if rising_edge(clk_250_mhz) then
			if (tx_state = TX_IDLE) and (tx_trigger = '1') then
				tx_signals_arr(00) <= tx_signal_00;
				tx_signals_arr(01) <= tx_signal_01;
				tx_signals_arr(02) <= tx_signal_02;
				tx_signals_arr(03) <= tx_signal_03;
				tx_signals_arr(04) <= tx_signal_04;
				tx_signals_arr(05) <= tx_signal_05;
				tx_signals_arr(06) <= tx_signal_06;
				tx_signals_arr(07) <= tx_signal_07;
				tx_signals_arr(08) <= tx_signal_08;
				tx_signals_arr(09) <= tx_signal_09;
				tx_signals_arr(10) <= tx_signal_10;
				tx_signals_arr(11) <= tx_signal_11;
				tx_signals_arr(12) <= tx_signal_12;
				tx_signals_arr(13) <= tx_signal_13;
				tx_signals_arr(14) <= tx_signal_14;
				tx_signals_arr(15) <= tx_signal_15;
				tx_signals_arr(16) <= tx_signal_16;
				tx_signals_arr(17) <= tx_signal_17;
				tx_signals_arr(18) <= tx_signal_18;
				tx_signals_arr(19) <= tx_signal_19;
				tx_signals_arr(20) <= tx_signal_20;
				tx_signals_arr(21) <= tx_signal_21;
				tx_signals_arr(22) <= tx_signal_22;
				tx_signals_arr(23) <= tx_signal_23;
				tx_signals_arr(24) <= tx_signal_24;
				tx_signals_arr(25) <= tx_signal_25;
				tx_signals_arr(26) <= tx_signal_26;
				tx_signals_arr(27) <= tx_signal_27;
				tx_signals_arr(28) <= tx_signal_28;
				tx_signals_arr(29) <= tx_signal_29;
				tx_signals_arr(30) <= tx_signal_30;
				tx_signals_arr(31) <= tx_signal_31;
				tx_signals_arr(32) <= tx_signal_32;
				tx_signals_arr(33) <= tx_signal_33;
				tx_signals_arr(34) <= tx_signal_34;
				tx_signals_arr(35) <= tx_signal_35;
				tx_signals_arr(36) <= tx_signal_36;
				tx_signals_arr(37) <= tx_signal_37;
				tx_signals_arr(38) <= tx_signal_38;
				tx_signals_arr(39) <= tx_signal_39;
				tx_signals_arr(40) <= tx_signal_40;
				tx_signals_arr(41) <= tx_signal_41;
				tx_signals_arr(42) <= tx_signal_42;
				tx_signals_arr(43) <= tx_signal_43;
				tx_signals_arr(44) <= tx_signal_44;
				tx_signals_arr(45) <= tx_signal_45;
				tx_signals_arr(46) <= tx_signal_46;
				tx_signals_arr(47) <= tx_signal_47;
				tx_signals_arr(48) <= tx_signal_48;
				tx_signals_arr(49) <= tx_signal_49;

				-- [flag F]

				if (tx_n_signals <= NUM_PORTS_TX) then
					tx_n_signals_i <= to_integer(tx_n_signals);
					tx_n_signals_i_is_odd <= tx_n_signals(0);
					tx_n_words_i <= to_integer(tx_n_signals);
				else
					tx_n_signals_i <= NUM_PORTS_TX;
					tx_n_signals_i_is_odd <= to_unsigned(NUM_PORTS_TX_IS_ODD, 1)(0);
					tx_n_words_i <= MAX_WORDS_TX;
				end if;

			end if;

			if (reset = '1') then
				tx_signals_arr <= (others => (others => '0'));
			end if;
		end if;
	end process;

	-- TX FSM
	process(clk_250_mhz) begin
		if rising_edge(clk_250_mhz) then
			case tx_state is
				when TX_IDLE =>
					if (tx_trigger = '1') and (tx_n_signals /= 0) then
						tx_state <= TX_TRANSMISSION;
					end if;

				when TX_TRANSMISSION =>
					if (tx_valid_transmission = '1') then
						if (tx_valid_last_transmission /= '1') then
							tx_word_index <= tx_word_index+1;
						else
							tx_state <= TX_IDLE;
							tx_word_index <= 0;
						end if;
					end if;

				when others =>
			end case;

			if (reset = '1') then
				tx_state <= TX_IDLE;
				tx_word_index <= 0;
			end if;
		end if;
	end process;

	-- Determine if the FSM is currently transmitting
	tx_in_transmission <= '1' when (tx_state = TX_TRANSMISSION) else '0';

	-- Determine if transmission was effective and last
	tx_valid_transmission <= tx_M_AXIS_TX_tvalid and M_AXIS_TX_tready;
	tx_valid_last_transmission <= tx_valid_transmission and tx_M_AXIS_TX_tlast;

	-- Determine if the current word is the last
	tx_last_word <= '1' when (tx_word_index >= tx_n_words_i-1) else '0';

	-- Set the tdata signal of the outgoing FIFO
	M_AXIS_TX_tdata <= tx_signals_arr(tx_word_index)
		when (tx_in_transmission = '1') else (others => '0');

	-- Set the tuser signal to 0 because not used
	M_AXIS_TX_tuser <= (others => '0');

	-- Set the internal tvalid and tlast signals
	tx_M_AXIS_TX_tvalid <= tx_in_transmission;
	tx_M_AXIS_TX_tlast <= tx_last_word;

	-- Set the external tvalid and tlast signals
	M_AXIS_TX_tvalid <= tx_M_AXIS_TX_tvalid;
	M_AXIS_TX_tlast <= tx_M_AXIS_TX_tlast;

	-----------------------------------------------
	-- RECEPTION LOGIC (RX)
	-----------------------------------------------

	-- Determine if transmission was effective and last
	rx_valid_transmission <= S_AXIS_RX_tvalid and r_S_AXIS_RX_tready;
	rx_valid_last_transmission <= rx_valid_transmission and S_AXIS_RX_tlast;

	-- Determine if max word index is exceeded
	rx_word_index_exceeding <= '1' when rx_word_index >= MAX_WORDS_RX else '0';

	-- Update the counter after each transmission
	process(clk_250_mhz) begin
		if rising_edge(clk_250_mhz) then
			if (rx_valid_transmission = '1') then
				if (rx_valid_last_transmission = '1') then
					rx_word_index <= 0;
				elsif (rx_word_index_exceeding = '0') then
					rx_word_index <= rx_word_index + 1;
				end if;
			end if;

			if (reset = '1') then
				rx_word_index <= 0;
			end if;
		end if;
	end process;

	-- Update the CRC check, payload size (as sent by the RT-Box)
	-- after last transmission
	process(clk_250_mhz) begin
		if rising_edge(clk_250_mhz) then
			if (rx_valid_last_transmission = '1') then
				rx_n_signals <= to_unsigned(rx_word_index+1, 16);
			end if;

			if (reset = '1') then
				rx_n_signals <= (others => '0');
			end if;
		end if;
	end process;

	-- Store data after each transmission
	process (clk_250_mhz) begin
		if rising_edge(clk_250_mhz) then
			if (rx_valid_transmission = '1') and (rx_word_index_exceeding = '0') then
				rx_signals_arr(rx_word_index) <= S_AXIS_RX_tdata;
			end if;

			if (reset = '1') then
				rx_signals_arr <= (others => (others => '0'));
			end if;
		end if;
	end process;

	-- Update outputs after the last transmission
	process (clk_250_mhz) begin
		if rising_edge(clk_250_mhz) then
			rx_valid_last_transmission_ff <= rx_valid_last_transmission;

			if (rx_valid_last_transmission_ff = '1') then
				rx_signal_00 <= rx_signals_arr(00);
				rx_signal_01 <= rx_signals_arr(01);
				rx_signal_02 <= rx_signals_arr(02);
				rx_signal_03 <= rx_signals_arr(03);
				rx_signal_04 <= rx_signals_arr(04);
				rx_signal_05 <= rx_signals_arr(05);
				rx_signal_06 <= rx_signals_arr(06);
				rx_signal_07 <= rx_signals_arr(07);
				rx_signal_08 <= rx_signals_arr(08);
				rx_signal_09 <= rx_signals_arr(09);
				rx_signal_10 <= rx_signals_arr(10);
				rx_signal_11 <= rx_signals_arr(11);
				rx_signal_12 <= rx_signals_arr(12);
				rx_signal_13 <= rx_signals_arr(13);
				rx_signal_14 <= rx_signals_arr(14);
				rx_signal_15 <= rx_signals_arr(15);
				rx_signal_16 <= rx_signals_arr(16);
				rx_signal_17 <= rx_signals_arr(17);
				rx_signal_18 <= rx_signals_arr(18);
				rx_signal_19 <= rx_signals_arr(19);
				rx_signal_20 <= rx_signals_arr(20);
				rx_signal_21 <= rx_signals_arr(21);
				rx_signal_22 <= rx_signals_arr(22);
				rx_signal_23 <= rx_signals_arr(23);
				rx_signal_24 <= rx_signals_arr(24);
				rx_signal_25 <= rx_signals_arr(25);
				rx_signal_26 <= rx_signals_arr(26);
				rx_signal_27 <= rx_signals_arr(27);
				rx_signal_28 <= rx_signals_arr(28);
				rx_signal_29 <= rx_signals_arr(29);
				rx_signal_30 <= rx_signals_arr(30);
				rx_signal_31 <= rx_signals_arr(31);
				rx_signal_32 <= rx_signals_arr(32);
				rx_signal_33 <= rx_signals_arr(33);
				rx_signal_34 <= rx_signals_arr(34);
				rx_signal_35 <= rx_signals_arr(35);
				rx_signal_36 <= rx_signals_arr(36);
				rx_signal_37 <= rx_signals_arr(37);
				rx_signal_38 <= rx_signals_arr(38);
				rx_signal_39 <= rx_signals_arr(39);
				rx_signal_40 <= rx_signals_arr(40);
				rx_signal_41 <= rx_signals_arr(41);
				rx_signal_42 <= rx_signals_arr(42);
				rx_signal_43 <= rx_signals_arr(43);
				rx_signal_44 <= rx_signals_arr(44);
				rx_signal_45 <= rx_signals_arr(45);
				rx_signal_46 <= rx_signals_arr(46);
				rx_signal_47 <= rx_signals_arr(47);
				rx_signal_48 <= rx_signals_arr(48);
				rx_signal_49 <= rx_signals_arr(49);

				-- [flag G]

			end if;

		end if;
	end process;

	-- Set the valid signal
	rx_vld <= rx_valid_last_transmission_ff;

	-- Set the internal tready signal (always ready)
	r_S_AXIS_RX_tready <= '1';

	-- Set the external tready signal (always ready)
	S_AXIS_RX_tready <= r_S_AXIS_RX_tready;

end Behavioral;
