#ifndef CONTROLLERS_H_
#define CONTROLLERS_H_

#include "DSP28x_Project.h"		// Headers for C28346 with useful definitions of data types


/**
 * Pseudo-object describing a PID controller
 * Definition and operation according to R.Longchamp's book...
 */
typedef struct{
	float32 kp,ki;				// Proportional and integral gains
	float32 limup;				// Upper saturation value of the output
	float32 limlow;				// Lower saturation value of the output
	Uint16 	N;					// Filtering parameter of the derivative
	float32 b;					// Offline-computed constants (contain the Ti and Td informations)
	float32 ui_prev;			// Previous value of the integral component
	float32 ud_prev;			// Previous value of the derivative component
	float32 e_prev;				// Previous value of the error
} PIDController;


/**
 * Pseudo-object describing a proportional-resonant controller (PR) with one resonant term
 * Definition and coding according to the following article:
 * "Proportional-resonant controllers and filters for grid-connected voltage-source converters"
 * from R. Teodorescu, F. Blaabjerg, M. Liserre and P.C. Loh
 * published in IEE Proc.-Electr. Power Appl., Vol. 153, N5, September 2006
 */
typedef struct{
	float32 kp;					// Proportional gain (the integral gain is contained in the other coefficients)
	float32 a1,a2,b0,b1,b2;		// Internal coefficients (computed offline)
	float32 ui_prev,ui_prev2;	// Previous values of the integral parts of the outputs (k-1, resp. k-2 samples)
	float32 e_prev,e_prev2;		// Previous values of the error
} PRController;


/**
 * Pseudo-object corresponding to a Maximum Power Point Tracking (MPPT) algorithm
 * This mainly contains the previous values of the power to be maximized as well as of the
 * quantity acting as a perturbation
 */
typedef struct{
	float32 power_prev;			// The previous value of the power extracted
	float32 meas_prev;			// The previous value of the acting quantity (e.g. the current)
	float32 reference_step;		// The size of the increment/decrement to be applied to the output
	float32 reference;			// The setpoint of the acting quantity
	float32 limup;				// Positive limit of the output (reference setpoint)
	float32	limlow;				// Negative limit of the output (reference setpoint)
	float32 iir_lpf;			// Low-passe filtering coefficient
} MPPTracker;


/**
 * Routine to configure the PID controller 'me' and pre-compute the necessary constants.
 * @param *me		the PID pseudo-object to be configured
 * @param kp		proportional gain
 * @param ki		integral gain
 * @param td 		derivative time-constant
 * @param limup		upper saturation threshold of the output quantity
 * @param limup		lower saturation threshold of the output quantity
 * @param tsample 	sampling (interrupt) time
 * @param N 		filtering factor of the derivative term (10 is a good typical value)
 * @return void
 */
void ConfigPIDController(PIDController* me, float32 kp, float32 ki, float32 td, float32 limup, float32 limlow, float32 tsample, Uint16 N);


/**
 * Routine to configure the PR controller 'me' and pre-compute the necessary constants.
 * @param *me		the PID pseudo-object to be configured
 * @param kp		proportional gain
 * @param ki		integral gain
 * @param td 		derivative time-constant
 * @param wres		center resonant frequency of the resonant term (in rad/s.)
 * @param wdamp		frequency "width" of the resonant term (limits the quality factor of the resonant term)  (in rad/s.)
 * @param tsample 	sampling (interrupt) time
 * @return void
 */
void ConfigPRController(PRController* me, float32 kp, float32 ki, float32 wres, float32 wdamp, float32 tsample);


/**
 * Routine to construct (i.e. initialize) the pseudo-object corresponding to a perturb-and-observe
 * MPPT algorithm
 * @param *me		the pseudo-object to initialize
 * @param ref_step	the size of the increment/decrement to apply at the output (e.g. on the current)
 * @param ref_init	the default/initial value of the output (the acting quantity, e.g. the current)
 * @param limup		the positive limit (maximum value) of the output
 * @param limlow	the negative limit (minimum value) of the output
 * @param iir_lpf	iir-type low-pass filtering coefficient
 */
void ConfigMPPTracker(MPPTracker* me, float32 ref_step, float32 ref_init, float32 limup, float32 limlow, float32 iir_lpf);


/**
 * Routines to run the pseudo-object 'me' depending of its actual nature (PI, PID, etc. controllers)
 * @param *me		the corresponding PID pseudo-object (parameters and state quantities)
 * @param error		the setpoint value minus the measured value
 * @return			the control variable for the measured quantity (output of the controller)
 */
float32 RunPIDController(PIDController* me, float32 error);
float32 RunPIController(PIDController* me, float32 error);
float32 RunIController(PIDController* me, float32 error);
float32 RunPController(PIDController* me, float32 error);


/**
 * Routines to run the pseudo-object 'me' for PR-like controllers
 * @param *me		the corresponding PR pseudo-object (parameters and state quantities)
 * @param error		the setpoint value minus the measured value
 * @return			the control variable for the measured quantity (output of the controller)
 */
float32 RunPRController(PRController* me, float32 error);


/**
 * Routine to run a Perturb&Observe Maximum Power Point Tracking (MPPT) algorithm
 * @param *me		the pseudo-object corresponding to the power to be maximized
 * @param measurement the perturbation quantity (e.g. the current)
 * @param power		the power to maximize
 * @return			the quantity acting as a perturbation (e.g. a current)
 */
float32 RunMPPTracking(MPPTracker* me, float32 measurement, float32 power);

#endif /*CONTROLLERS_H_*/
