/*
 *  @title    User-level main file: TEMPLATE TO BE MODIFIED !
 *	@author   imperix Ltd (dev@imperix.ch)
 *	@file     user.cpp
 *	@comment  Core functions require that this file implements at least these functions :
 *			  USER_SAFE UserInit(void)
 *			  void UserError(int code);
 */

#include "Driver/peripherals.h"													// Available DSP peripherals
#include "Core/interrupts.h"													// Interrupt-related routines
#include "user.h"																// This file's header.


/**
 * Global variables
 */
USER_STATE current_mode = STANDBY;
USER_STATE next_mode = STANDBY;


/**
 * The user should put here his/her initialization routines.
 * This function is called only once at startup.
 */
USER_SAFE UserInit(void)
{
	/************************************************************************************
	 * Configuration of the frequency generators and PWM channels
	 * --------------------------
	 * - SetFreqGenPeriod(freqgen, period) is used to configure the needed switching
	 * 	 frequency. The 'period' parameter defines the period in clock ticks of the FPGA.
	 * - Up to 4 frequency generators can be used simultaneously.
	 * - ConfigPWMChannel(channel, freqgen, style, deadtime) is used to configure the
	 * 	 modulation and deadtime parameters
	 * - SetPWMPhase(channel, phase) computes the corrected phase parameter depending on
	 *	 the configured deadtime.
	 ************************************************************************************/

	SetFreqGenPeriod(FREQGEN0, (int)(SWITCHING_PERIOD/FPGA_CLK_PERIOD));		// 1 kHz switching frequency is set on frequency generator #0
	ConfigPWMChannel(FREQGEN0, 0, TRIANGLE, (int)(300e-9/FPGA_CLK_PERIOD));		// Channel #0 is mapped to frequency generator #0 with a triangular carrier and a 300 ns deadtime
	SetPWMPhase(0, 0.0);														// Channel #0 is configured with a phase of zero with respect to the frequency generator
	//ActivatePWMChannel(0);													// Channel #0 is activated (will produce an output when the boombox is enabled)


	/************************************************************************************
	 * Configuration of the ADC channels
	 * ------------------------
	 * - The configuration of the analog front-end (filter, gains, safety threshold and
	 * 	 high/low impedance) must be configured on the frontpanel (rotary button) first.
	 * - The full-span resolution is 16bits corresponding to [-10V;+10V]
	 * - Software gains and offset adjustments can be applied using the function
	 *   SetADCAdjustments(channel, gain, offset) for calibration purposes.
	 ************************************************************************************/

	SetADCAdjustments(0, 1.0, 0.0);												// Analog input #0 with default gain (1.0) and offset (0.0)


	/************************************************************************************
	 * Configuration of the main user interrupts
	 * ------------------------
	 * - Three interrupts sources are available, from highest to lowest priority:
	 *	 A. External event #1 (from FPGA, line XINT1)
	 *		Can be activated and linked to an event by calling RegisterExt1Interrupt().
	 *	 B. External event #2 (from FPGA, line XINT2)
	 *		Can be activated and linked to an event by calling RegisterExt2Interrupt().
	 *	 C. Timer-driven
	 * 		Can be activated and configured when calling RegisterTimerInterrupt(...).
	 * - During the configuration of an interrupt line, the user must specify the
	 *   interrupt service routine (user-ISR) that will be serviced. Example:
	 *   RegisterExt1Interrupt(&MyUserISR, source, phase, postscaler);
	 ************************************************************************************/	
	 
	RegisterExt1Interrupt(&UserInterrupt1, FREQGEN0, 0.0, 0);					// The ISR (Interrupt Service Routine) is set to the user function UserInterrupt2
																				// Ext1 is clocked on frequency generator #0 (configured previously with a period of 1 kHz)
																				// with a relative phase of 0.0 and no postscaling


	/************************************************************************************
	 * Configuration of sampling clock
	 * ------------------------
	 * - The sampling clock defines the instant at which measurements are sampled and the
	 *   ADC conversion starts. This process is done entirely by hardware without
	 *   triggering any DSP interrupts.
	 * - ConfigSampling(freqgen, phase) is used to configure the sampling clock.
	 *   It maps the sampling clock to one of the previously configured frequency
	 *   generators and adds the desired phase. It also takes into account a 4 us advance,
	 *   corresponding to the conversion and acquisition time. This way, if an interrupt
	 *   and the sampling clock are configured on the same frequency generator with the
	 *   same phase, the data read in the ISR is guaranteed to be the last value sampled.
	 *   This doesn't, however, prevent the user from configuring different phases for the
	 *   sampling clock and the interrupt which is responsible for reading the ADC values.
	 ************************************************************************************/
	
	ConfigSampling(FREQGEN0, 0.5);												// The sampling is configured in the middle of the switching period of frequency generator #0


	/************************************************************************************
	 * User-defined initialization routines
	 ************************************************************************************/
	current_mode = STANDBY;														// Initialize the application to its STANDBY state
	return SAFE;																// Return SAFE if no error
}


/**
 * First user-level interrupt service routine.
 * Remember that it has the highest level of priority when mapped to Ext1, intermediate when
 * mapped to Ext2 and lowest when mapped to the CPU Timer.
 */
USER_SAFE UserInterrupt1(void)
{
	// Switch depending on the operation mode (state machine)
	switch (current_mode){
	   	case STANDBY:															// Typically used as long as the user doesn't start the converter
	   					next_mode = STANDBY;
						break;

	   	case STARTUP:  															// Typically used to progressively energize the application
	   																			// A condition could make jump to normal operation afterwards
	   					next_mode = STARTUP;
						break;

	   	case NORMAL:															// Typically used during normal operation (other modes could be defined)
	   					next_mode = NORMAL;

	   					SetPWMDutyCycle(0, 0.75);								// The duty cycle of PWM channel #0 is set to 0.75
	   																			// Its switching frequency is that of its related frequency generator, i.e. #0 with 1kHz.
	   																			// It's carrier waveform, phase and deadtime were previously set using ConfigPWMChannel(...)

	   					break;

	   	case SHUTDOWN:															// Typically used to properly de-energize the application (safe shutdown)
	   					next_mode = SHUTDOWN;
	   					break;

	   	case EMERGENCY:															// Typically used during emergency shut down
	   					next_mode = EMERGENCY;
	   					break;
	}


	/************************************************************************************
	 * Update of the FPGA parameters
	 * ------------------------
	 * At the end of the control interrupt, all FPGA-based peripherals should have their
	 * parameters coherently updated. Among others, the following updates are made:
	 * - All frequency generator parameters (needed for variable-frequency operation)
	 * - All PWM parameters (such as duty-cycles or phase parameters)
	 ************************************************************************************/

	UpdatePWMData();															// All realtime FPGA data are updated

	current_mode = next_mode;													// Switch to the next state
    return SAFE;																// Return a SAFE status (or UNSAFE otherwise)
}


/**
 * Second user-level interrupt service routine.
 * Remember that it has the highest level of priority when mapped to Ext1, intermediate when
 * mapped to Ext2 and lowest when mapped to the CPU Timer.
 */
USER_SAFE UserInterrupt2(void)
{
	return SAFE;
}


/*
 * This routine aims to handle critical errors by requesting an appropriate answer at the user level.
 * Typically, the application should be blocked and the core will wait for manual clearing of the error.
 */
void UserError(ERROR_SOURCE source)
{
	current_mode = EMERGENCY;													// Typically used to de-energize the application
}
