#include "user.h"

/**
 * Global variables
 * float, int, and unsigned int, can be monitored from imperix Cockpit
 */
float Vdc_meas  = 0; // Measured DC bus voltage

/**
 * Initialisation routine executed only once, before the first call of the main interrupt
 * To be used to configure all needed peripherals and perform all needed initialisations
 */
tUserSafe UserInit(void)
{
	/**
	 * Configuration of the main interrupt:
	 * - The frequency of CLOCK_0 is set to the desired switching frequency
	 * - The main interrupt routine is mapped on CLOCK_0, with a phase of 0.5.
	 * - As the the sampling of the ADC channels happens just before the interrupt is executed,
	 *   setting a phase of 0.5 allows to sample in the middle of each switching periods.
	 */
	Clock_SetFrequency(CLOCK_0, SW_FREQ);
	ConfigureMainInterrupt(UserInterrupt, CLOCK_0, 0.5);

	/**
	 * Configuration of the ADC channels, with:
	 * - sensor sensitivity defined in user.h
	 * - no sensor offset compensation
	 */
	Adc_ConfigureInput(ADC0, V_GAIN, 0.0);

	/**
	 * Configuration of PWM channel 0 (lines 0 & 1)
	 * - mapping on CLOCK_0
	 * - triangle carrier
	 * - 1 microsecond dead-time between complementary signals
	 */
	CbPwm_ConfigureChannel(PWM_CHANNEL_0, CLOCK_0, TRIANGLE, 1e-6);

	/**
	 * Note: the function CbPwm_ConfigureChannel() is a helper defined in user.h
	 * By default this routine:
	 * - sets a phase of 0 between CLOCK_0 and PWM carrier
	 * - sets outputs as 2 complementary signals
	 * - activates the PWM channel
	 */

	return SAFE;
}


/**
 * Main interrupt routine
 */
tUserSafe UserInterrupt(void)
{
	// Retrieve the measurements
	Vdc_meas  = Adc_GetValue(ADC0);

	// Update the PWM duty-cycles
	CbPwm_SetDutyCycle(PWM_CHANNEL_0, 0.75);

	return SAFE;
}


/**
 * Routine executed when the core state goes into FAULT mode
 */
void UserError(tErrorSource source)
{

}
